<?php
require_once '../../config/config.php';

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn() || $user->getRole() !== 'user') {
    header('Location: ../../index.php');
    exit;
}

// Get user data
$userId = $user->getId();
$userData = $user->getCurrentUser();

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Get user's PINs
$sql = "SELECT p.*, 
        pt.transfer_date as last_transfer_date,
        u.username as transferred_by_username,
        u.full_name as transferred_by_name
        FROM pins p 
        LEFT JOIN pin_transfers pt ON p.id = pt.pin_id AND pt.to_user = ?
        LEFT JOIN users u ON pt.from_user = u.id
        WHERE p.current_owner = ? 
        ORDER BY p.created_at DESC";
$stmt = $connection->prepare($sql);
$stmt->bind_param("ii", $userId, $userId);
$stmt->execute();
$pins = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PIN Management - Zealmart</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        body { background: #e5e5e5; }
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(180deg, #3a86ff 0%, #8338ec 100%);
            padding-top: 20px;
            border-top-right-radius: 20px;
            border-bottom-right-radius: 20px;
            box-shadow: 2px 0 12px rgba(51,51,51,0.08);
            transition: background 0.3s;
        }
        .sidebar .sidebar-brand {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 2rem;
        }
        .sidebar .sidebar-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: #fff;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.5rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .sidebar .sidebar-logo i {
            font-size: 2rem;
            color: #3a86ff;
        }
        .sidebar .sidebar-title {
            color: #fff;
            font-weight: bold;
            font-size: 1.2rem;
            letter-spacing: 1px;
        }
        .sidebar-section {
            color: #e0e1dd;
            font-size: 0.85rem;
            text-transform: uppercase;
            margin: 1.5rem 0 0.5rem 1rem;
            letter-spacing: 0.08em;
        }
        .sidebar a {
            color: #fff;
            padding: 12px 22px;
            display: flex;
            align-items: center;
            border-radius: 8px 0 0 8px;
            margin-bottom: 6px;
            font-size: 1.05rem;
            transition: background 0.2s, color 0.2s, box-shadow 0.2s;
        }
        .sidebar a.active, .sidebar a:hover {
            background: rgba(255, 255, 255, 0.15);
            color: #ffbe0b;
            box-shadow: 2px 2px 8px rgba(0,0,0,0.07);
            font-weight: bold;
        }
        .sidebar a i {
            margin-right: 12px;
            font-size: 1.2em;
        }
        @media (max-width: 767.98px) {
            .sidebar {
                position: fixed;
                z-index: 1050;
                left: 0;
                top: 0;
                width: 220px;
                height: 100vh;
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                border-radius: 0 20px 20px 0;
            }
            .sidebar.collapse.show {
                transform: translateX(0);
            }
            .main-content {
                padding: 10px;
            }
        }
        .main-content {
            background: #fff;
            min-height: 100vh;
            padding: 2rem 2rem 2rem 2rem;
        }
    </style>
</head>
<body>
<div class="container-fluid">
    <div class="row">
        <!-- Sidebar Overlay (for mobile) -->
        <div id="sidebarOverlay" style="display:none;position:fixed;top:0;left:0;width:100vw;height:100vh;background:rgba(0,0,0,0.3);z-index:1049;"></div>
        <!-- Sidebar Toggle Button (visible on mobile) -->
        <button class="btn btn-dark d-md-none m-2" type="button" data-toggle="collapse" data-target="#userSidebar" aria-controls="userSidebar" aria-expanded="false" aria-label="Toggle navigation">
            <i class="fas fa-bars"></i> Menu
        </button>
        <?php include_once '../../includes/user_sidebar.php'; ?>
        <main class="col-md-10 ml-sm-auto main-content">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">PIN Management</h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <button type="button" class="btn btn-sm btn-primary me-2" data-toggle="modal" data-target="#transferPinModal">
                        <i class="fas fa-share"></i> Transfer PINs
                    </button>
                    <button type="button" class="btn btn-sm btn-success" data-toggle="modal" data-target="#activatePinModal">
                        <i class="fas fa-check"></i> Activate PIN
                    </button>
                </div>
            </div>

            <!-- PIN Statistics -->
            <div class="row mb-4">
                <div class="col-md-4">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Total PINs</h5>
                            <p class="card-text h3"><?php echo count($pins); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Available PINs</h5>
                            <p class="card-text h3"><?php 
                                echo count(array_filter($pins, function($pin) {
                                    return $pin['status'] !== 'activated';
                                }));
                            ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Used PINs</h5>
                            <p class="card-text h3"><?php 
                                echo count(array_filter($pins, function($pin) {
                                    return $pin['status'] === 'activated';
                                }));
                            ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- PINs List -->
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead>
                        <tr>
                            <th><input type="checkbox" id="selectAll" class="form-check-input"></th>
                            <th>PIN Number</th>
                            <th>Batch ID</th>
                            <th>Status</th>
                            <th>Transferred By</th>
                            <th>Transfer Date</th>
                            <th>Created Date</th>
                            <th>Activated Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($pins)): ?>
                        <tr>
                            <td colspan="8" class="text-center">No PINs found</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($pins as $pin): ?>
                        <tr>
                            <td>
                                <?php if ($pin['status'] !== 'activated'): ?>
                                <input type="checkbox" class="form-check-input pin-checkbox" value="<?php echo $pin['id']; ?>">
                                <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars($pin['pin_number']); ?></td>
                            <td><?php echo htmlspecialchars($pin['batch_id']); ?></td>
                            <td>
                                <span class="badge bg-<?php 
                                    echo match($pin['status']) {
                                        'created' => 'secondary',
                                        'transferred' => 'primary',
                                        'activated' => 'success',
                                        default => 'secondary'
                                    };
                                ?>">
                                    <?php echo ucfirst($pin['status']); ?>
                                </span>
                            </td>
                            <td><?php echo $pin['transferred_by_name'] ? htmlspecialchars($pin['transferred_by_name']) . ' (' . htmlspecialchars($pin['transferred_by_username']) . ')' : '-'; ?></td>
                            <td><?php echo $pin['last_transfer_date'] ? date('Y-m-d H:i', strtotime($pin['last_transfer_date'])) : '-'; ?></td>
                            <td><?php echo date('Y-m-d H:i', strtotime($pin['created_at'])); ?></td>
                            <td><?php echo $pin['activated_at'] ? date('Y-m-d H:i', strtotime($pin['activated_at'])) : '-'; ?></td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
</div>

<!-- Transfer PIN Modal -->
<div class="modal fade" id="transferPinModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Transfer PINs</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
            </div>
            <div class="modal-body">
                <form id="transferPinForm">
                    <div class="mb-3">
                        <label for="username" class="form-label">Recipient Username</label>
                        <input type="text" class="form-control" id="username" name="username" required>
                        <div class="form-text">Enter the username of the person you want to transfer PINs to.</div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Selected PINs</label>
                        <div id="selectedPins" class="alert alert-info">
                            No PINs selected
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="submitTransfer">Transfer PINs</button>
            </div>
        </div>
    </div>
</div>

<!-- Activate PIN Modal -->
<div class="modal fade" id="activatePinModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Activate PIN</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
            </div>
            <div class="modal-body">
                <form id="activatePinForm">
                    <div class="mb-3">
                        <label for="pin_number" class="form-label">PIN Number</label>
                        <select class="form-select" id="pin_number" name="pin_number" required>
                            <option value="">Select a PIN</option>
                            <?php foreach ($pins as $pin): ?>
                                <?php if ($pin['status'] !== 'activated'): ?>
                                <option value="<?php echo htmlspecialchars($pin['id']); ?>" data-pin-number="<?php echo htmlspecialchars($pin['pin_number']); ?>">
                                    <?php echo htmlspecialchars($pin['pin_number']); ?> (<?php echo htmlspecialchars($pin['batch_id']); ?>)
                                </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                        <div class="form-text">Select an available PIN to activate.</div>
                    </div>

                    <div class="mb-3">
                        <label for="search_user" class="form-label">Search User</label>
                        <div class="input-group">
                            <input type="text" class="form-control" id="search_user" 
                                   placeholder="Search by username, name, email, or phone">
                            <button class="btn btn-outline-secondary" type="button" id="searchUserBtn">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                        <div class="form-text">Search for an inactive user to activate.</div>
                    </div>

                    <div class="mb-3">
                        <label for="target_user" class="form-label">Select User</label>
                        <select class="form-select" id="target_user" name="target_user" required disabled>
                            <option value="">First search for a user</option>
                        </select>
                        <div class="form-text">Select the user you want to activate.</div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-success" id="submitActivation">Activate PIN</button>
            </div>
        </div>
    </div>
</div>

<!-- Include footer -->
<?php include_once '../../includes/footer.php'; ?>

<script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
$(function() {
    var sidebar = $('#userSidebar');
    var overlay = $('#sidebarOverlay');
    sidebar.on('show.bs.collapse', function() {
        if (window.innerWidth < 768) overlay.show();
    });
    sidebar.on('hide.bs.collapse', function() {
        overlay.hide();
    });
    overlay.on('click', function() {
        sidebar.collapse('hide');
    });
    $(window).on('resize', function() {
        if (window.innerWidth >= 768) overlay.hide();
    });
});
</script>

<style>
.card {
    border-radius: 0.5rem;
    box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
}

.card-title {
    color: #6c757d;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.1em;
}

.card-text.h3 {
    margin-bottom: 0;
    color: #212529;
}

.table th {
    font-weight: 500;
    text-transform: uppercase;
    font-size: 0.75rem;
    letter-spacing: 0.1em;
}
</style>
</body>
</html> 