<?php
require_once '../config/config.php';

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn() || $user->getRole() !== 'user') {
    header('Location: ../index.php');
    exit;
}

// Get user data
$userId = $user->getId();
$userData = $user->getCurrentUser();

// Get genealogy data
$db = Database::getInstance();
$connection = $db->getConnection();

// Get direct referrals (level 1)
$query = "SELECT u.id, u.username, u.full_name, u.status, g.created_at, g.placement_level, g.position FROM genealogy g JOIN users u ON g.user_id = u.id WHERE g.parent_id = ? ORDER BY g.placement_level, g.position";
$stmt = $connection->prepare($query);
$stmt->bind_param("i", $userId);
$stmt->execute();
$directReferrals = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// For each direct referral, get their direct referrals (level 2)
$tree = [];
foreach ($directReferrals as $ref1) {
    $ref1_id = $ref1['id'];
    $query2 = "SELECT u.id, u.username, u.full_name, u.status, g.created_at, g.placement_level, g.position FROM genealogy g JOIN users u ON g.user_id = u.id WHERE g.parent_id = ? ORDER BY g.placement_level, g.position";
    $stmt2 = $connection->prepare($query2);
    $stmt2->bind_param("i", $ref1_id);
    $stmt2->execute();
    $level2 = $stmt2->get_result()->fetch_all(MYSQLI_ASSOC);
    $tree[] = [
        'user' => $ref1,
        'children' => $level2
    ];
}

// Get referral link
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$referralLink = $protocol . '://' . $_SERVER['HTTP_HOST'] . SITE_URL . '/public/register.php?ref=' . $userData['username'];


// Recursive function to fetch genealogy tree
function fetchGenealogyTree($db, $user_id, $max_depth = 3, $current_depth = 1) {
    if ($current_depth > $max_depth) return [];
    $sql = "SELECT u.id, u.username, u.full_name, u.status FROM genealogy g JOIN users u ON g.user_id = u.id WHERE g.parent_id = ? ORDER BY g.position";
    $stmt = $db->getConnection()->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $children = [];
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) {
        $row['children'] = fetchGenealogyTree($db, $row['id'], $max_depth, $current_depth + 1);
        $children[] = $row;
    }
    return $children;
}

// Recursive function to render the tree
function renderGenealogyTree($user, $children) {
    ?>
    <li>
        <div class="tree-node">
            <img src="../assets/images/avatar.png" class="avatar" alt="avatar">
            <div class="username"><?php echo htmlspecialchars($user['username']); ?></div>
        </div>
        <?php if (!empty($children)): ?>
        <ul>
            <?php foreach ($children as $child): ?>
                <?php renderGenealogyTree($child, $child['children']); ?>
            <?php endforeach; ?>
        </ul>
        <?php endif; ?>
    </li>
    <?php
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Genealogy - Zealmart</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        body { background: #e5e5e5; }
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(180deg, #3a86ff 0%, #8338ec 100%);
            padding-top: 20px;
            border-top-right-radius: 20px;
            border-bottom-right-radius: 20px;
            box-shadow: 2px 0 12px rgba(51,51,51,0.08);
            transition: background 0.3s;
        }
        .sidebar .sidebar-brand {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 2rem;
        }
        .sidebar .sidebar-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: #fff;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.5rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .sidebar .sidebar-logo i {
            font-size: 2rem;
            color: #3a86ff;
        }
        .sidebar .sidebar-title {
            color: #fff;
            font-weight: bold;
            font-size: 1.2rem;
            letter-spacing: 1px;
        }
        .sidebar-section {
            color: #e0e1dd;
            font-size: 0.85rem;
            text-transform: uppercase;
            margin: 1.5rem 0 0.5rem 1rem;
            letter-spacing: 0.08em;
        }
        .sidebar a {
            color: #fff;
            padding: 12px 22px;
            display: flex;
            align-items: center;
            border-radius: 8px 0 0 8px;
            margin-bottom: 6px;
            font-size: 1.05rem;
            transition: background 0.2s, color 0.2s, box-shadow 0.2s;
        }
        .sidebar a.active, .sidebar a:hover {
            background: rgba(255, 255, 255, 0.15);
            color: #ffbe0b;
            box-shadow: 2px 2px 8px rgba(0,0,0,0.07);
            font-weight: bold;
        }
        .sidebar a i {
            margin-right: 12px;
            font-size: 1.2em;
        }
        @media (max-width: 767.98px) {
            .sidebar {
                position: fixed;
                z-index: 1050;
                left: 0;
                top: 0;
                width: 220px;
                height: 100vh;
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                border-radius: 0 20px 20px 0;
            }
            .sidebar.collapse.show {
                transform: translateX(0);
            }
            .main-content {
                padding: 10px;
            }
        }
        .main-content {
            background: #fff;
            min-height: 100vh;
            padding: 2rem 2rem 2rem 2rem;
        }
    </style>
</head>
<body>
<div class="container-fluid">
    <div class="row">
        <!-- Sidebar Overlay (for mobile) -->
        <div id="sidebarOverlay" style="display:none;position:fixed;top:0;left:0;width:100vw;height:100vh;background:rgba(0,0,0,0.3);z-index:1049;"></div>
        <!-- Sidebar Toggle Button (visible on mobile) -->
        <button class="btn btn-dark d-md-none m-2" type="button" data-toggle="collapse" data-target="#userSidebar" aria-controls="userSidebar" aria-expanded="false" aria-label="Toggle navigation">
            <i class="fas fa-bars"></i> Menu
        </button>
        <?php include_once '../includes/user_sidebar.php'; ?>
        <main class="col-md-10 ml-sm-auto main-content">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Network Genealogy</h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <button type="button" class="btn btn-sm btn-primary" id="copyReferralLink">
                        <i class="fas fa-link"></i> Copy Referral Link
                    </button>
                </div>
            </div>

            <!-- Referral Link Section -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Your Referral Link</h5>
                            <div class="input-group">
                                <input type="text" class="form-control" id="referralLinkInput" value="<?php echo $referralLink; ?>" readonly>
                                <button class="btn btn-outline-secondary" type="button" onclick="copyReferralLink()">
                                    <i class="fas fa-copy"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tree View -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Network Tree</h5>
                            <ul class="genealogy-tree">
                                <?php renderGenealogyTree($userData, fetchGenealogyTree($db, $userId)); ?>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Referral List -->
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Direct Referrals</h5>
                            <div class="table-responsive">
                                <table class="table table-striped table-sm">
                                    <thead>
                                        <tr>
                                            <th>Username</th>
                                            <th>Full Name</th>
                                            <th>Level</th>
                                            <th>Position</th>
                                            <th>Status</th>
                                            <th>Joined Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($directReferrals as $referral): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($referral['username']); ?></td>
                                            <td><?php echo htmlspecialchars($referral['full_name']); ?></td>
                                            <td><?php echo $referral['placement_level']; ?></td>
                                            <td><?php echo $referral['position']; ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $referral['status'] === 'active' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($referral['status']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo date('Y-m-d', strtotime($referral['created_at'])); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>
<script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
$(function() {
    var sidebar = $('#userSidebar');
    var overlay = $('#sidebarOverlay');
    sidebar.on('show.bs.collapse', function() {
        if (window.innerWidth < 768) overlay.show();
    });
    sidebar.on('hide.bs.collapse', function() {
        overlay.hide();
    });
    overlay.on('click', function() {
        sidebar.collapse('hide');
    });
    $(window).on('resize', function() {
        if (window.innerWidth >= 768) overlay.hide();
    });
});
</script>
<style>
.genealogy-tree, .genealogy-tree ul {
    padding-top: 20px; position: relative;
    display: flex; justify-content: center;
    list-style: none;
}
.genealogy-tree ul { padding-top: 40px; }
.genealogy-tree li {
    text-align: center;
    position: relative;
    padding: 20px 5px 0 5px;
}
.genealogy-tree li::before, .genealogy-tree li::after {
    content: '';
    position: absolute; top: 0; right: 50%;
    border-top: 2px solid #ccc; width: 50%; height: 20px;
}
.genealogy-tree li::after {
    right: auto; left: 50%; border-left: 2px solid #ccc;
}
.genealogy-tree li:only-child::before, .genealogy-tree li:only-child::after {
    display: none;
}
.genealogy-tree li:only-child { padding-top: 0; }
.genealogy-tree li:first-child::before, .genealogy-tree li:last-child::after {
    border: 0 none;
}
.tree-node {
    display: flex; flex-direction: column; align-items: center;
}
.avatar {
    width: 60px; height: 60px; border-radius: 50%; border: 2px solid #6c63ff;
    background: #fff;
}
.username {
    margin-top: 8px; font-weight: bold; color: #6c63ff;
    background: #f3f0ff; padding: 2px 8px; border-radius: 6px;
}
</style> 