<?php
class User {
    private $db;
    private $user_data;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->user_data = null;
    }

    public function register($full_name, $email, $phone, $password, $ref_username = null) {
        // Generate unique username
        $username = $this->generateUsername();
        
        // Hash password
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        $sql = "INSERT INTO users (username, full_name, email, phone, password) 
                VALUES (?, ?, ?, ?, ?)";
        
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bind_param("sssss", $username, $full_name, $email, $phone, $hashed_password);
        
        if ($stmt->execute()) {
            $user_id = $this->db->getLastId();
            $parent_id = null;
            if ($ref_username) {
                $sql2 = "SELECT id FROM users WHERE username = ?";
                $stmt2 = $this->db->getConnection()->prepare($sql2);
                $stmt2->bind_param("s", $ref_username);
                $stmt2->execute();
                $result2 = $stmt2->get_result();
                if ($result2 && $row2 = $result2->fetch_assoc()) {
                    $parent_id = $row2['id'];
                }
            }
            return ['success' => true, 'username' => $username, 'user_id' => $user_id, 'parent_id' => $parent_id];
        }
        
        return ['success' => false, 'error' => $this->db->getLastError()];
    }

    public function login($username, $password) {
        $sql = "SELECT * FROM users WHERE username = ?";
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();
            if (password_verify($password, $user['password'])) {
                $this->user_data = $user;
                $this->initializeSession();
                return true;
            }
        }
        return false;
    }

    public function logout() {
        session_destroy();
        $this->user_data = null;
        return true;
    }

    public function resetPassword($email) {
        $sql = "SELECT id FROM users WHERE email = ?";
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            // Generate reset token and send email
            // TODO: Implement email sending functionality
            return true;
        }
        return false;
    }

    private function generateUsername() {
        $sql = "SELECT MAX(CAST(SUBSTRING(username, 4, 5) AS UNSIGNED)) as last_num 
                FROM users WHERE username LIKE 'ZEL%'";
        $result = $this->db->query($sql);
        $row = $result->fetch_assoc();
        
        $next_num = ($row['last_num'] ?? 24999) + 1;
        return sprintf("ZEL%05dA", $next_num);
    }

    private function initializeSession() {
        $_SESSION['user_id'] = $this->user_data['id'];
        $_SESSION['username'] = $this->user_data['username'];
        $_SESSION['role'] = $this->user_data['role'];
        $_SESSION['last_activity'] = time();
    }

    public function isLoggedIn() {
        return isset($_SESSION['user_id']);
    }

    public function isAdmin() {
        return isset($_SESSION['role']) && $_SESSION['role'] === 'admin';
    }

    public function getCurrentUser() {
        if ($this->user_data === null && $this->isLoggedIn()) {
            $sql = "SELECT * FROM users WHERE id = ?";
            $stmt = $this->db->getConnection()->prepare($sql);
            $stmt->bind_param("i", $_SESSION['user_id']);
            $stmt->execute();
            $result = $stmt->get_result();
            $this->user_data = $result->fetch_assoc();
        }
        return $this->user_data;
    }

    public function getId() {
        if ($this->isLoggedIn()) {
            return $_SESSION['user_id'];
        }
        return null;
    }

    public function getUsername() {
        if ($this->isLoggedIn()) {
            if (isset($_SESSION['username'])) {
                return $_SESSION['username'];
            } else {
                $user = $this->getCurrentUser();
                return $user['username'] ?? null;
            }
        }
        return null;
    }

    public function getRole() {
        if ($this->isLoggedIn()) {
            if (isset($_SESSION['role'])) {
                return $_SESSION['role'];
            } else {
                $user = $this->getCurrentUser();
                return $user['role'] ?? null;
            }
        }
        return null;
    }
}
?> 