<?php
require_once '../../config/config.php';

// Set headers
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Invalid request method']);
    exit;
}

// Get POST data
$pan_number = trim($_POST['pan_number'] ?? '');
$id_proof_type = trim($_POST['id_proof_type'] ?? '');
$id_proof_number = trim($_POST['id_proof_number'] ?? '');

// Validate input
if (empty($pan_number) || empty($id_proof_type) || empty($id_proof_number)) {
    echo json_encode(['success' => false, 'error' => 'All fields are required']);
    exit;
}

// Validate PAN format (ABCDE1234F)
if (!preg_match('/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/', $pan_number)) {
    echo json_encode(['success' => false, 'error' => 'Invalid PAN number format']);
    exit;
}

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Begin transaction
$connection->begin_transaction();

try {
    // Check if user already has KYC details
    $sql = "SELECT id FROM user_profiles WHERE user_id = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        // Update existing KYC details
        $sql = "UPDATE user_profiles 
                SET pan_number = ?, 
                    id_proof_type = ?, 
                    id_proof_number = ?,
                    kyc_status = 'pending'
                WHERE user_id = ?";
        $stmt = $connection->prepare($sql);
        $stmt->bind_param("sssi", $pan_number, $id_proof_type, $id_proof_number, $_SESSION['user_id']);
    } else {
        // Insert new KYC details
        $sql = "INSERT INTO user_profiles (user_id, pan_number, id_proof_type, id_proof_number, kyc_status) 
                VALUES (?, ?, ?, ?, 'pending')";
        $stmt = $connection->prepare($sql);
        $stmt->bind_param("isss", $_SESSION['user_id'], $pan_number, $id_proof_type, $id_proof_number);
    }

    if ($stmt->execute()) {
        $connection->commit();
        echo json_encode(['success' => true, 'message' => 'KYC details updated successfully']);
    } else {
        throw new Exception('Failed to update KYC details');
    }
} catch (Exception $e) {
    $connection->rollback();
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 