<?php
require_once '../../config/config.php';

// Set headers
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Invalid request method']);
    exit;
}

// Get POST data
$bank_name = trim($_POST['bank_name'] ?? '');
$account_number = trim($_POST['account_number'] ?? '');
$account_type = trim($_POST['account_type'] ?? '');
$ifsc_code = trim($_POST['ifsc_code'] ?? '');
$branch_name = trim($_POST['branch_name'] ?? '');
$upi_id = trim($_POST['upi_id'] ?? '');
$upi_number = trim($_POST['upi_number'] ?? '');
$upi_app = trim($_POST['upi_app'] ?? '');

// Validate required fields
if (empty($bank_name) || empty($account_number) || empty($account_type) || empty($ifsc_code) || empty($branch_name)) {
    echo json_encode(['success' => false, 'error' => 'Bank details are required']);
    exit;
}

// Validate IFSC code format (AAAA0123456)
if (!preg_match('/^[A-Z]{4}0[A-Z0-9]{6}$/', $ifsc_code)) {
    echo json_encode(['success' => false, 'error' => 'Invalid IFSC code format']);
    exit;
}

// Validate account number (basic check)
if (!preg_match('/^\d{9,18}$/', $account_number)) {
    echo json_encode(['success' => false, 'error' => 'Invalid account number format']);
    exit;
}

// Validate UPI ID format if provided
if (!empty($upi_id) && !preg_match('/^[\w\.\-]+@[\w\-]+$/', $upi_id)) {
    echo json_encode(['success' => false, 'error' => 'Invalid UPI ID format']);
    exit;
}

// Validate UPI number if provided
if (!empty($upi_number) && !preg_match('/^\d{10}$/', $upi_number)) {
    echo json_encode(['success' => false, 'error' => 'Invalid UPI number format']);
    exit;
}

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Begin transaction
$connection->begin_transaction();

try {
    // Check if user already has banking details
    $sql = "SELECT id FROM banking_details WHERE user_id = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        // Update existing banking details
        $sql = "UPDATE banking_details 
                SET bank_name = ?, 
                    account_number = ?, 
                    account_type = ?,
                    ifsc_code = ?,
                    branch_name = ?,
                    upi_id = ?,
                    upi_number = ?,
                    upi_app = ?
                WHERE user_id = ?";
        $stmt = $connection->prepare($sql);
        $stmt->bind_param("ssssssssi", 
            $bank_name, 
            $account_number, 
            $account_type,
            $ifsc_code,
            $branch_name,
            $upi_id,
            $upi_number,
            $upi_app,
            $_SESSION['user_id']
        );
    } else {
        // Insert new banking details
        $sql = "INSERT INTO banking_details 
                (user_id, bank_name, account_number, account_type, ifsc_code, branch_name, upi_id, upi_number, upi_app) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $connection->prepare($sql);
        $stmt->bind_param("issssssss", 
            $_SESSION['user_id'],
            $bank_name, 
            $account_number, 
            $account_type,
            $ifsc_code,
            $branch_name,
            $upi_id,
            $upi_number,
            $upi_app
        );
    }

    if ($stmt->execute()) {
        $connection->commit();
        echo json_encode(['success' => true, 'message' => 'Banking details updated successfully']);
    } else {
        throw new Exception('Failed to update banking details');
    }
} catch (Exception $e) {
    $connection->rollback();
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 