<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Prevent any output before our JSON response
ob_start();

// Disable error display
ini_set('display_errors', 0);
error_reporting(0);

// Set JSON header
header('Content-Type: application/json');

// Define root path
define('ROOT_PATH', realpath(dirname(__FILE__) . '/../../'));

// Include required files
require_once ROOT_PATH . '/config/config.php';
require_once ROOT_PATH . '/classes/Database.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Function to send JSON response and exit
function sendJsonResponse($success, $message, $error = null) {
    // Clear any output buffers
    while (ob_get_level()) {
        ob_end_clean();
    }
    
    $response = ['success' => $success];
    if ($success) {
        $response['message'] = $message;
    } else {
        $response['error'] = $error ?? $message;
    }
    echo json_encode($response);
    exit;
}

try {
    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        throw new Exception('User not logged in');
    }

    // Check if it's a POST request
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method');
    }

    // Get POST data
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';

    // Validate input
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        throw new Exception('All fields are required');
    }

    if ($new_password !== $confirm_password) {
        throw new Exception('New passwords do not match');
    }

    if (strlen($new_password) < MIN_PASSWORD_LENGTH) {
        throw new Exception('Password must be at least ' . MIN_PASSWORD_LENGTH . ' characters long');
    }

    // Get database instance
    $db = Database::getInstance();
    $connection = $db->getConnection();

    if (!$connection) {
        throw new Exception("Database connection failed");
    }

    // Get user's current password
    $stmt = $connection->prepare("SELECT password FROM users WHERE id = ?");
    if (!$stmt) {
        throw new Exception("Failed to prepare statement: " . $connection->error);
    }

    $stmt->bind_param("i", $_SESSION['user_id']);
    if (!$stmt->execute()) {
        throw new Exception("Failed to execute statement: " . $stmt->error);
    }

    $result = $stmt->get_result();
    $user = $result->fetch_assoc();

    if (!$user) {
        throw new Exception('User not found');
    }

    // Verify current password
    if (!password_verify($current_password, $user['password'])) {
        throw new Exception('Current password is incorrect');
    }

    // Hash new password
    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);

    // Update password
    $stmt = $connection->prepare("UPDATE users SET password = ? WHERE id = ?");
    if (!$stmt) {
        throw new Exception("Failed to prepare update statement: " . $connection->error);
    }

    $stmt->bind_param("si", $hashed_password, $_SESSION['user_id']);
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to update password: " . $stmt->error);
    }

    if ($stmt->affected_rows > 0) {
        sendJsonResponse(true, 'Password updated successfully');
    } else {
        throw new Exception("No changes were made to the password");
    }

} catch (Exception $e) {
    // Log the error
    error_log("Password change error: " . $e->getMessage());
    sendJsonResponse(false, 'An error occurred while changing password', $e->getMessage());
} 