<?php
require_once '../../config/config.php';

// Set headers
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Invalid request method']);
    exit;
}

// Get POST data
$ticketId = $_POST['ticket_id'] ?? 0;
$message = trim($_POST['message'] ?? '');

// Validate required fields
if (!$ticketId || empty($message)) {
    echo json_encode(['success' => false, 'error' => 'Ticket ID and message are required']);
    exit;
}

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Begin transaction
$connection->begin_transaction();

try {
    // Check if ticket exists and belongs to user
    $sql = "SELECT id, status FROM support_tickets WHERE id = ? AND user_id = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("ii", $ticketId, $_SESSION['user_id']);
    $stmt->execute();
    $ticket = $stmt->get_result()->fetch_assoc();

    if (!$ticket) {
        throw new Exception('Ticket not found');
    }

    if ($ticket['status'] === 'closed' || $ticket['status'] === 'deleted') {
        throw new Exception('Cannot reply to a closed or deleted ticket');
    }

    // Insert reply message
    $sql = "INSERT INTO ticket_messages (ticket_id, sender_id, message) VALUES (?, ?, ?)";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("iis", $ticketId, $_SESSION['user_id'], $message);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to save reply');
    }

    // Update ticket status to active if it was just created
    if ($ticket['status'] === 'created') {
        $sql = "UPDATE support_tickets SET status = 'active' WHERE id = ?";
        $stmt = $connection->prepare($sql);
        $stmt->bind_param("i", $ticketId);
        
        if (!$stmt->execute()) {
            throw new Exception('Failed to update ticket status');
        }
    }

    // Commit transaction
    $connection->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Reply sent successfully'
    ]);

} catch (Exception $e) {
    $connection->rollback();
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 