<?php
require_once '../../config/config.php';

// Set headers
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Check if ticket ID is provided
$ticketId = $_GET['ticket_id'] ?? 0;
if (!$ticketId) {
    echo json_encode(['success' => false, 'error' => 'Ticket ID is required']);
    exit;
}

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

try {
    // Get ticket details
    $sql = "SELECT t.*, u.username, u.full_name 
            FROM support_tickets t 
            JOIN users u ON t.user_id = u.id 
            WHERE t.id = ? AND t.user_id = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("ii", $ticketId, $_SESSION['user_id']);
    $stmt->execute();
    $ticket = $stmt->get_result()->fetch_assoc();

    if (!$ticket) {
        echo json_encode(['success' => false, 'error' => 'Ticket not found']);
        exit;
    }

    // Get ticket messages
    $sql = "SELECT m.*, u.username, u.full_name, u.role 
            FROM ticket_messages m 
            JOIN users u ON m.sender_id = u.id 
            WHERE m.ticket_id = ? 
            ORDER BY m.created_at ASC";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("i", $ticketId);
    $stmt->execute();
    $messages = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

    // Build HTML response
    $html = '
    <div class="ticket-header mb-4">
        <h6>Ticket ID: ' . htmlspecialchars($ticket['ticket_id']) . '</h6>
        <p class="mb-1"><strong>Subject:</strong> ' . htmlspecialchars($ticket['subject']) . '</p>
        <p class="mb-1"><strong>Status:</strong> 
            <span class="badge bg-' . match($ticket['status']) {
                'created' => 'secondary',
                'active' => 'primary',
                'in_process' => 'info',
                'closed' => 'success',
                'inactive' => 'warning',
                'deleted' => 'danger',
                default => 'secondary'
            } . '">' . ucfirst(str_replace('_', ' ', $ticket['status'])) . '</span>
        </p>
        <p class="mb-1"><strong>Created:</strong> ' . date('Y-m-d H:i', strtotime($ticket['created_at'])) . '</p>
    </div>
    <div class="ticket-messages">';

    foreach ($messages as $message) {
        $isAdmin = $message['role'] === 'admin';
        $html .= '
        <div class="ticket-message ' . ($isAdmin ? 'admin' : 'user') . '">
            <div class="ticket-meta">
                <strong>' . htmlspecialchars($message['full_name']) . '</strong>
                <span class="ms-2">' . date('Y-m-d H:i', strtotime($message['created_at'])) . '</span>
            </div>
            <div class="message-content mt-2">
                ' . nl2br(htmlspecialchars($message['message'])) . '
            </div>
        </div>';
    }

    $html .= '</div>';

    echo json_encode([
        'success' => true,
        'html' => $html,
        'ticket' => $ticket
    ]);

} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 