<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../../config/config.php';

// Set headers
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Invalid request method']);
    exit;
}

// Get POST data
$subject = trim($_POST['subject'] ?? '');
$message = trim($_POST['message'] ?? '');

// Validate required fields
if (empty($subject) || empty($message)) {
    echo json_encode(['success' => false, 'error' => 'Subject and message are required']);
    exit;
}

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Begin transaction
$connection->begin_transaction();

try {
    // Generate ticket ID (Format: TID0225001)
    $sql = "SELECT MAX(CAST(SUBSTRING(ticket_id, 4) AS UNSIGNED)) as last_id FROM support_tickets";
    $result = $connection->query($sql);
    $row = $result->fetch_assoc();
    $lastId = $row['last_id'] ?? 0;
    $newId = $lastId + 1;
    $ticketId = 'TID' . str_pad($newId, 6, '0', STR_PAD_LEFT);

    // Insert ticket
    $sql = "INSERT INTO support_tickets (ticket_id, user_id, subject, status) VALUES (?, ?, ?, 'created')";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("sis", $ticketId, $_SESSION['user_id'], $subject);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to create ticket');
    }

    $ticketDbId = $stmt->insert_id;

    // Insert initial message
    $sql = "INSERT INTO ticket_messages (ticket_id, sender_id, message) VALUES (?, ?, ?)";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("iis", $ticketDbId, $_SESSION['user_id'], $message);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to save ticket message');
    }

    // Commit transaction
    $connection->commit();
    
    echo json_encode([
        'success' => true, 
        'message' => 'Ticket created successfully',
        'ticket_id' => $ticketId
    ]);

} catch (Exception $e) {
    $connection->rollback();
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 