<?php
require_once '../../config/config.php';

// Set headers
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Invalid request method']);
    exit;
}

// Get POST data
$username = trim($_POST['username'] ?? '');
$pinIds = $_POST['pins'] ?? [];

// Validate required fields
if (empty($username) || empty($pinIds)) {
    echo json_encode(['success' => false, 'error' => 'Username and PINs are required']);
    exit;
}

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Begin transaction
$connection->begin_transaction();

try {
    // Get recipient user
    $sql = "SELECT id FROM users WHERE username = ? AND status = 'active'";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $recipient = $stmt->get_result()->fetch_assoc();

    if (!$recipient) {
        throw new Exception('Recipient user not found or inactive');
    }

    // Verify PIN ownership and status
    $pinIdsStr = implode(',', array_map('intval', $pinIds));
    $sql = "SELECT id FROM pins WHERE id IN ($pinIdsStr) AND current_owner = ? AND status != 'activated'";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $validPins = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

    if (count($validPins) !== count($pinIds)) {
        throw new Exception('Some PINs are invalid, already activated, or do not belong to you');
    }

    // Update PIN ownership and status
    $sql = "UPDATE pins SET current_owner = ?, status = 'transferred' WHERE id = ?";
    $updateStmt = $connection->prepare($sql);

    // Insert transfer records
    $sql = "INSERT INTO pin_transfers (pin_id, from_user, to_user) VALUES (?, ?, ?)";
    $transferStmt = $connection->prepare($sql);

    foreach ($pinIds as $pinId) {
        // Update PIN
        $updateStmt->bind_param("ii", $recipient['id'], $pinId);
        if (!$updateStmt->execute()) {
            throw new Exception('Failed to update PIN ownership');
        }

        // Record transfer
        $transferStmt->bind_param("iii", $pinId, $_SESSION['user_id'], $recipient['id']);
        if (!$transferStmt->execute()) {
            throw new Exception('Failed to record PIN transfer');
        }
    }

    // Commit transaction
    $connection->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'PINs transferred successfully'
    ]);

} catch (Exception $e) {
    $connection->rollback();
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 