<?php
require_once '../../config/config.php';

// Set headers
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Invalid request method']);
    exit;
}

// Get POST data
$pinNumber = trim($_POST['pin_number'] ?? '');
$targetUserId = intval($_POST['target_user'] ?? 0);

// Validate required fields
if (empty($pinNumber) || empty($targetUserId)) {
    echo json_encode(['success' => false, 'error' => 'PIN number and target user are required']);
    exit;
}

// Get database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Begin transaction
$connection->begin_transaction();

try {
    // Check if PIN exists and is available
    $sql = "SELECT id, current_owner, status FROM pins WHERE pin_number = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("s", $pinNumber);
    $stmt->execute();
    $pin = $stmt->get_result()->fetch_assoc();

    if (!$pin) {
        throw new Exception('PIN not found');
    }

    if ($pin['status'] === 'activated') {
        throw new Exception('PIN has already been activated');
    }

    if ($pin['current_owner'] !== $_SESSION['user_id']) {
        throw new Exception('This PIN does not belong to you');
    }

    // Check if target user exists and is inactive
    $sql = "SELECT id, status, username, full_name FROM users WHERE id = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("i", $targetUserId);
    $stmt->execute();
    $targetUser = $stmt->get_result()->fetch_assoc();

    if (!$targetUser) {
        throw new Exception('Target user not found');
    }

    if ($targetUser['status'] === 'active') {
        throw new Exception('Target user is already activated');
    }

    // Update PIN status
    $sql = "UPDATE pins SET status = 'activated', activated_at = CURRENT_TIMESTAMP WHERE id = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("i", $pin['id']);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to activate PIN');
    }

    // Activate target user account
    $sql = "UPDATE users SET status = 'active' WHERE id = ?";
    $stmt = $connection->prepare($sql);
    $stmt->bind_param("i", $targetUserId);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to activate user account');
    }

    // MLM Earnings Distribution Logic (same as admin)
    $mlm_plan = [
        1 => ['members' => 3, 'payout' => 250],
        2 => ['members' => 9, 'payout' => 500],
        3 => ['members' => 27, 'payout' => 1000],
        4 => ['members' => 81, 'payout' => 2000],
        5 => ['members' => 243, 'payout' => 4000],
        6 => ['members' => 729, 'payout' => 8000],
        7 => ['members' => 2187, 'payout' => 16000],
        8 => ['members' => 6561, 'payout' => 32000],
        9 => ['members' => 19683, 'payout' => 64000],
        10 => ['members' => 59049, 'payout' => 128000],
        11 => ['members' => 177147, 'payout' => 256000],
        12 => ['members' => 531441, 'payout' => 512000],
        13 => ['members' => 1594323, 'payout' => 1024000],
        14 => ['members' => 4782969, 'payout' => 2048000],
        15 => ['members' => 14348907, 'payout' => 4096000],
        16 => ['members' => 43046721, 'payout' => 8192000],
        17 => ['members' => 129140163, 'payout' => 16384000],
        18 => ['members' => 387420489, 'payout' => 32768000],
        19 => ['members' => 1162261467, 'payout' => 65536000],
        20 => ['members' => 3486784401, 'payout' => 131072000],
    ];
    function get_downline_at_level($connection, $user_id, $level) {
        $current_level = 0;
        $current_ids = [$user_id];
        while ($current_level < $level) {
            if (empty($current_ids)) return [];
            $placeholders = implode(',', array_fill(0, count($current_ids), '?'));
            $types = str_repeat('i', count($current_ids));
            $stmt = $connection->prepare("SELECT user_id FROM genealogy WHERE parent_id IN ($placeholders)");
            $stmt->bind_param($types, ...$current_ids);
            $stmt->execute();
            $result = $stmt->get_result();
            $next_ids = [];
            while ($row = $result->fetch_assoc()) {
                $next_ids[] = $row['user_id'];
            }
            $current_ids = $next_ids;
            $current_level++;
        }
        return $current_ids;
    }
    $current_user_id = $targetUserId;
    for ($level = 1; $level <= 20; $level++) {
        // Get parent
        $sql = "SELECT parent_id FROM genealogy WHERE user_id = ?";
        $stmt = $connection->prepare($sql);
        if (!$stmt) break;
        $stmt->bind_param("i", $current_user_id);
        if (!$stmt->execute()) break;
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        if (!$row || !$row['parent_id']) break;
        $upline_id = $row['parent_id'];
        // Check if upline qualifies for payout at this level
        $downline = get_downline_at_level($connection, $upline_id, $level);
        if (count($downline) >= $mlm_plan[$level]['members']) {
            // Check if earning already exists
            $check = $connection->prepare("SELECT id FROM earnings WHERE user_id = ? AND amount = ? AND earning_type = 'cash'");
            $check->bind_param("id", $upline_id, $mlm_plan[$level]['payout']);
            $check->execute();
            $check->store_result();
            if ($check->num_rows == 0) {
                // Insert earning
                $ins = $connection->prepare("INSERT INTO earnings (user_id, amount, earning_type, status) VALUES (?, ?, 'cash', 'available')");
                $ins->bind_param("id", $upline_id, $mlm_plan[$level]['payout']);
                $ins->execute();
            }
        }
        $current_user_id = $upline_id;
    }
    // END MLM Earnings Distribution Logic

    // Commit transaction
    $connection->commit();
    
    echo json_encode([
        'success' => true,
        'message' => sprintf(
            'PIN activated successfully and account for %s (%s) has been activated',
            $targetUser['full_name'],
            $targetUser['username']
        )
    ]);

} catch (Exception $e) {
    $connection->rollback();
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 