<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get ticket ID from URL
$ticketId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$ticketId) {
    header('Location: index.php');
    exit;
}

// Initialize database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Get ticket details
$query = "SELECT st.*, u.username, u.full_name, u.email, u.phone,
                 DATE_FORMAT(st.created_at, '%Y-%m-%d %H:%i') as formatted_date
          FROM support_tickets st
          JOIN users u ON st.user_id = u.id
          WHERE st.id = ?";
$stmt = $connection->prepare($query);
$stmt->bind_param('i', $ticketId);
$stmt->execute();
$ticket = $stmt->get_result()->fetch_assoc();

if (!$ticket) {
    header('Location: index.php');
    exit;
}

// Get ticket messages
$query = "SELECT tm.*, u.username, u.full_name, u.role,
                 DATE_FORMAT(tm.created_at, '%Y-%m-%d %H:%i') as formatted_date
          FROM ticket_messages tm
          JOIN users u ON tm.sender_id = u.id
          WHERE tm.ticket_id = ?
          ORDER BY tm.created_at ASC";
$stmt = $connection->prepare($query);
$stmt->bind_param('i', $ticketId);
$stmt->execute();
$messages = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get ticket status history
$statusLogs = [];
try {
    $query = "SELECT tsl.*, u.username,
                    DATE_FORMAT(tsl.changed_at, '%Y-%m-%d %H:%i') as formatted_date
            FROM ticket_status_logs tsl
            JOIN users u ON tsl.changed_by = u.id
            WHERE tsl.ticket_id = ?
            ORDER BY tsl.changed_at DESC";
    $stmt = $connection->prepare($query);
    $stmt->bind_param('i', $ticketId);
    $stmt->execute();
    $statusLogs = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
} catch (Exception $e) {
    // If table doesn't exist, we'll just show an empty status history
    error_log("Error fetching status logs: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - View Ticket</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
        .message {
            margin-bottom: 20px;
            padding: 15px;
            border-radius: 10px;
        }
        .message.admin {
            background-color: #e9ecef;
            margin-left: 50px;
        }
        .message.user {
            background-color: #f8f9fa;
            margin-right: 50px;
        }
        .message-header {
            margin-bottom: 10px;
            font-size: 0.9rem;
        }
        .message-content {
            white-space: pre-wrap;
        }
        .status-timeline {
            position: relative;
            padding-left: 30px;
        }
        .status-timeline::before {
            content: '';
            position: absolute;
            left: 15px;
            top: 0;
            bottom: 0;
            width: 2px;
            background-color: #dee2e6;
        }
        .status-item {
            position: relative;
            margin-bottom: 15px;
        }
        .status-item::before {
            content: '';
            position: absolute;
            left: -30px;
            top: 5px;
            width: 10px;
            height: 10px;
            border-radius: 50%;
            background-color: #007bff;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="../customers.php"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="../pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="../reports/"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="index.php" class="active"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-8">
                        <h2>
                            <a href="index.php" class="btn btn-outline-primary mr-2">
                                <i class="fas fa-arrow-left"></i>
                            </a>
                            Ticket #<?php echo htmlspecialchars($ticket['ticket_id']); ?>
                        </h2>
                    </div>
                    <div class="col-md-4 text-right">
                        <?php if ($ticket['status'] === 'closed'): ?>
                            <form id="reopenForm" class="form-inline justify-content-end">
                                <input type="hidden" name="ticket_id" value="<?php echo $ticket['id']; ?>">
                                <input type="hidden" name="status" value="active">
                                <button type="submit" class="btn btn-warning">
                                    <i class="fas fa-redo mr-2"></i> Reopen Ticket
                                </button>
                            </form>
                        <?php else: ?>
                            <form id="updateStatusForm" class="form-inline justify-content-end">
                                <input type="hidden" name="ticket_id" value="<?php echo $ticket['id']; ?>">
                                <select name="status" class="form-control mr-2">
                                    <option value="created" <?php echo $ticket['status'] === 'created' ? 'selected' : ''; ?>>Created</option>
                                    <option value="active" <?php echo $ticket['status'] === 'active' ? 'selected' : ''; ?>>Active</option>
                                    <option value="in_process" <?php echo $ticket['status'] === 'in_process' ? 'selected' : ''; ?>>In Process</option>
                                    <option value="closed" <?php echo $ticket['status'] === 'closed' ? 'selected' : ''; ?>>Closed</option>
                                </select>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save mr-2"></i> Update Status
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>

                    <!-- Add status badge -->
                    <div class="col-12 mt-3">
                        <span class="badge badge-<?php 
                            echo $ticket['status'] === 'closed' ? 'secondary' : 
                                ($ticket['status'] === 'active' ? 'success' : 
                                    ($ticket['status'] === 'in_process' ? 'info' : 'primary')); 
                        ?>">
                            Status: <?php echo ucfirst(str_replace('_', ' ', $ticket['status'])); ?>
                        </span>
                    </div>
                </div>

                <div class="row">
                    <!-- Ticket Details and Messages -->
                    <div class="col-md-8">
                        <!-- Ticket Information -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="mb-0">Ticket Information</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <p><strong>Subject:</strong> <?php echo htmlspecialchars($ticket['subject']); ?></p>
                                        <p><strong>Status:</strong> 
                                            <span class="badge badge-<?php 
                                                echo $ticket['status'] === 'created' ? 'primary' : 
                                                    ($ticket['status'] === 'active' ? 'success' : 
                                                    ($ticket['status'] === 'in_process' ? 'info' : 
                                                    ($ticket['status'] === 'closed' ? 'secondary' : 'warning'))); 
                                            ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $ticket['status'])); ?>
                                            </span>
                                        </p>
                                        <p><strong>Created:</strong> <?php echo $ticket['formatted_date']; ?></p>
                                    </div>
                                    <div class="col-md-6">
                                        <p><strong>User:</strong> <?php echo htmlspecialchars($ticket['username']); ?></p>
                                        <p><strong>Name:</strong> <?php echo htmlspecialchars($ticket['full_name']); ?></p>
                                        <p><strong>Email:</strong> <?php echo htmlspecialchars($ticket['email']); ?></p>
                                        <p><strong>Phone:</strong> <?php echo htmlspecialchars($ticket['phone']); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Messages -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="mb-0">Messages</h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($messages)): ?>
                                    <p class="text-center text-muted">No messages yet</p>
                                <?php else: ?>
                                    <?php foreach ($messages as $message): ?>
                                        <div class="message <?php echo $message['role'] === 'admin' ? 'admin' : 'user'; ?>">
                                            <div class="message-header">
                                                <strong><?php echo htmlspecialchars($message['full_name']); ?></strong>
                                                <span class="text-muted ml-2"><?php echo $message['formatted_date']; ?></span>
                                            </div>
                                            <div class="message-content">
                                                <?php echo nl2br(htmlspecialchars($message['message'])); ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Reply Form -->
                        <?php if ($ticket['status'] !== 'closed'): ?>
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="mb-0">Reply</h5>
                                </div>
                                <div class="card-body">
                                    <form id="replyForm">
                                        <input type="hidden" name="ticket_id" value="<?php echo $ticket['id']; ?>">
                                        <div class="form-group">
                                            <textarea name="message" class="form-control" rows="5" required></textarea>
                                        </div>
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-paper-plane mr-2"></i> Send Reply
                                        </button>
                                    </form>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Status Timeline -->
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Status Timeline</h5>
                            </div>
                            <div class="card-body">
                                <div class="status-timeline">
                                    <?php foreach ($statusLogs as $log): ?>
                                        <div class="status-item">
                                            <div class="text-muted small"><?php echo $log['formatted_date']; ?></div>
                                            <div>
                                                Status changed to 
                                                <strong><?php echo ucfirst(str_replace('_', ' ', $log['status'])); ?></strong>
                                            </div>
                                            <div class="small">by <?php echo htmlspecialchars($log['username']); ?></div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        $(document).ready(function() {
            // Handle status update form submission
            $('#updateStatusForm').on('submit', function(e) {
                e.preventDefault();
                
                if ($(this).find('select[name="status"]').val() === 'closed') {
                    if (!confirm('Are you sure you want to close this ticket? You can reopen it later if needed.')) {
                        return;
                    }
                }
                
                submitStatusUpdate($(this));
            });

            // Handle reopen form submission
            $('#reopenForm').on('submit', function(e) {
                e.preventDefault();
                
                if (confirm('Are you sure you want to reopen this ticket?')) {
                    submitStatusUpdate($(this));
                }
            });

            function submitStatusUpdate($form) {
                $.ajax({
                    url: 'update_status.php',
                    method: 'POST',
                    data: $form.serialize(),
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            window.location.reload();
                        } else {
                            alert(response.message || 'Failed to update ticket status');
                        }
                    },
                    error: function() {
                        alert('An error occurred while updating the ticket status');
                    }
                });
            }

            // Handle reply form submission
            $('#replyForm').on('submit', function(e) {
                e.preventDefault();
                
                $.ajax({
                    url: 'reply.php',
                    method: 'POST',
                    data: $(this).serialize(),
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            window.location.reload();
                        } else {
                            alert(response.message || 'Failed to send reply');
                        }
                    },
                    error: function() {
                        alert('An error occurred while sending the reply');
                    }
                });
            });
        });
    </script>
</body>
</html> 