<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to JSON
header('Content-Type: application/json');

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get POST data
$ticketId = isset($_POST['ticket_id']) ? (int)$_POST['ticket_id'] : 0;
$status = isset($_POST['status']) ? trim($_POST['status']) : '';

// Validate input
if (!$ticketId || !$status) {
    echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
    exit;
}

// Validate status value
$validStatuses = ['created', 'active', 'in_process', 'closed', 'inactive', 'deleted'];
if (!in_array($status, $validStatuses)) {
    echo json_encode(['success' => false, 'message' => 'Invalid status value']);
    exit;
}

try {
    // Get database connection
    $db = Database::getInstance();
    $connection = $db->getConnection();

    // Start transaction
    $connection->begin_transaction();

    // Get current ticket and user information
    $query = "SELECT t.status, t.user_id FROM support_tickets t WHERE t.id = ?";
    $stmt = $connection->prepare($query);
    $stmt->bind_param('i', $ticketId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        throw new Exception('Ticket not found');
    }
    
    $ticketInfo = $result->fetch_assoc();
    $currentStatus = $ticketInfo['status'];
    
    // Check if user has permission to update the ticket
    if (!$user->isAdmin() && $ticketInfo['user_id'] !== $user->getId()) {
        throw new Exception('You do not have permission to update this ticket');
    }

    // If ticket is closed and being reopened
    if ($currentStatus === 'closed' && $status !== 'closed') {
        // Only allow reopening to 'active' status
        if ($status !== 'active') {
            throw new Exception('Closed tickets can only be reopened to active status');
        }
    }

    // Update ticket status
    $query = "UPDATE support_tickets SET status = ?, updated_at = NOW() WHERE id = ?";
    $stmt = $connection->prepare($query);
    $stmt->bind_param('si', $status, $ticketId);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to update ticket status');
    }

    // Log status change
    $query = "INSERT INTO ticket_status_logs (ticket_id, status, changed_by, changed_at) 
              VALUES (?, ?, ?, NOW())";
    $stmt = $connection->prepare($query);
    $userId = $user->getId();
    $stmt->bind_param('isi', $ticketId, $status, $userId);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to log status change: ' . $stmt->error);
    }

    // Commit transaction
    $connection->commit();

    // Prepare success message
    $message = $currentStatus === 'closed' && $status === 'active' 
        ? 'Ticket reopened successfully' 
        : 'Ticket status updated successfully';

    echo json_encode([
        'success' => true,
        'message' => $message
    ]);

} catch (Exception $e) {
    // Rollback transaction on error
    if (isset($connection)) {
        $connection->rollback();
    }

    // Log error
    error_log("Error updating ticket status: " . $e->getMessage());

    echo json_encode([
        'success' => false,
        'message' => 'Failed to update ticket status: ' . $e->getMessage()
    ]);
} 