<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to JSON
header('Content-Type: application/json');

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get POST data
$ticketId = isset($_POST['ticket_id']) ? (int)$_POST['ticket_id'] : 0;
$message = isset($_POST['message']) ? trim($_POST['message']) : '';

// Validate input
if (!$ticketId || !$message) {
    echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
    exit;
}

try {
    // Get database connection
    $db = Database::getInstance();
    $connection = $db->getConnection();

    // Start transaction
    $connection->begin_transaction();

    // Check if ticket exists and is not closed
    $query = "SELECT status FROM support_tickets WHERE id = ?";
    $stmt = $connection->prepare($query);
    $stmt->bind_param('i', $ticketId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        throw new Exception('Ticket not found');
    }
    
    $ticket = $result->fetch_assoc();
    if ($ticket['status'] === 'closed') {
        throw new Exception('Cannot reply to a closed ticket');
    }

    // Add the reply message
    $query = "INSERT INTO ticket_messages (ticket_id, sender_id, message, created_at) 
              VALUES (?, ?, ?, NOW())";
    $stmt = $connection->prepare($query);
    $adminId = $user->getId();
    $stmt->bind_param('iis', $ticketId, $adminId, $message);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to add reply message');
    }

    // Update ticket status to in_process if it's not already
    if ($ticket['status'] === 'created' || $ticket['status'] === 'active') {
        $query = "UPDATE support_tickets SET status = 'in_process', updated_at = NOW() WHERE id = ?";
        $stmt = $connection->prepare($query);
        $stmt->bind_param('i', $ticketId);
        
        if (!$stmt->execute()) {
            throw new Exception('Failed to update ticket status');
        }

        // Log status change
        $query = "INSERT INTO ticket_status_logs (ticket_id, status, changed_by, changed_at) 
                  VALUES (?, 'in_process', ?, NOW())";
        $stmt = $connection->prepare($query);
        $stmt->bind_param('ii', $ticketId, $adminId);
        
        if (!$stmt->execute()) {
            throw new Exception('Failed to log status change');
        }
    }

    // Commit transaction
    $connection->commit();

    echo json_encode([
        'success' => true,
        'message' => 'Reply sent successfully'
    ]);

} catch (Exception $e) {
    // Rollback transaction on error
    if (isset($connection)) {
        $connection->rollback();
    }

    // Log error
    error_log("Error sending reply: " . $e->getMessage());

    echo json_encode([
        'success' => false,
        'message' => 'Failed to send reply: ' . $e->getMessage()
    ]);
} 