<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get filter parameters
$status = $_GET['status'] ?? '';
$search = trim($_GET['search'] ?? '');
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 10;
$offset = ($page - 1) * $limit;

// Initialize database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Build base query
$baseQuery = "FROM support_tickets st 
             JOIN users u ON st.user_id = u.id 
             WHERE 1=1";

$params = [];
$types = "";

if ($status) {
    $baseQuery .= " AND st.status = ?";
    $params[] = $status;
    $types .= "s";
}

if ($search) {
    $baseQuery .= " AND (st.ticket_id LIKE ? OR u.username LIKE ? OR u.full_name LIKE ? OR st.subject LIKE ?)";
    $params = array_merge($params, ["%$search%", "%$search%", "%$search%", "%$search%"]);
    $types .= "ssss";
}

// Get total records for pagination
$countQuery = "SELECT COUNT(*) as total " . $baseQuery;
$stmt = $connection->prepare($countQuery);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$totalRecords = $stmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalRecords / $limit);

// Get tickets with pagination
$query = "SELECT st.*, u.username, u.full_name,
                 DATE_FORMAT(st.created_at, '%Y-%m-%d %H:%i') as formatted_date,
                 (SELECT COUNT(*) FROM ticket_messages WHERE ticket_id = st.id) as message_count
          " . $baseQuery . "
          ORDER BY st.created_at DESC 
          LIMIT ? OFFSET ?";

$params[] = $limit;
$params[] = $offset;
$types .= "ii";

$stmt = $connection->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$tickets = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get ticket statistics
$statsQuery = "SELECT 
                COALESCE(COUNT(*), 0) as total_tickets,
                COALESCE(SUM(CASE WHEN status = 'created' THEN 1 ELSE 0 END), 0) as new_tickets,
                COALESCE(SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END), 0) as active_tickets,
                COALESCE(SUM(CASE WHEN status = 'in_process' THEN 1 ELSE 0 END), 0) as in_process_tickets,
                COALESCE(SUM(CASE WHEN status = 'closed' THEN 1 ELSE 0 END), 0) as closed_tickets
              FROM support_tickets";
$stats = $connection->query($statsQuery)->fetch_assoc();

// Helper function to safely format numbers
function safeNumberFormat($value) {
    return number_format(intval($value ?? 0));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Support Tickets</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(180deg, #3a86ff 0%, #8338ec 100%);
            padding-top: 20px;
            border-top-right-radius: 20px;
            border-bottom-right-radius: 20px;
            box-shadow: 2px 0 12px rgba(51,51,51,0.08);
            transition: background 0.3s;
        }
        .sidebar .sidebar-brand {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 2rem;
        }
        .sidebar .sidebar-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: #fff;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.5rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .sidebar .sidebar-logo i {
            font-size: 2rem;
            color: #3a86ff;
        }
        .sidebar .sidebar-title {
            color: #fff;
            font-weight: bold;
            font-size: 1.2rem;
            letter-spacing: 1px;
        }
        .sidebar-section {
            color: #e0e1dd;
            font-size: 0.85rem;
            text-transform: uppercase;
            margin: 1.5rem 0 0.5rem 1rem;
            letter-spacing: 0.08em;
        }
        .sidebar a {
            color: #fff;
            padding: 12px 22px;
            display: flex;
            align-items: center;
            border-radius: 8px 0 0 8px;
            margin-bottom: 6px;
            font-size: 1.05rem;
            transition: background 0.2s, color 0.2s, box-shadow 0.2s;
        }
        .sidebar a.active, .sidebar a:hover {
            background: rgba(255, 255, 255, 0.15);
            color: #ffbe0b;
            box-shadow: 2px 2px 8px rgba(0,0,0,0.07);
            font-weight: bold;
        }
        .sidebar a i {
            margin-right: 12px;
            font-size: 1.2em;
        }
        @media (max-width: 767.98px) {
            .sidebar {
                position: fixed;
                z-index: 1050;
                left: 0;
                top: 0;
                width: 220px;
                height: 100vh;
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                border-radius: 0 20px 20px 0;
            }
            .sidebar.collapse.show {
                transform: translateX(0);
            }
            .main-content {
                padding: 10px;
            }
        }
        .main-content {
            padding: 20px;
        }
        .stats-card {
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .stats-card i {
            font-size: 2rem;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar Overlay (for mobile) -->
            <div id="sidebarOverlay" style="display:none;position:fixed;top:0;left:0;width:100vw;height:100vh;background:rgba(0,0,0,0.3);z-index:1049;"></div>
            <!-- Sidebar Toggle Button (visible on mobile) -->
            <button class="btn btn-dark d-md-none m-2" type="button" data-toggle="collapse" data-target="#adminSidebar" aria-controls="adminSidebar" aria-expanded="false" aria-label="Toggle navigation">
                <i class="fas fa-bars"></i> Menu
            </button>
            <!-- Sidebar -->
            <nav id="adminSidebar" class="col-md-2 sidebar collapse d-md-block">
                <div class="sidebar-brand">
                    <div class="sidebar-logo">
                        <i class="fas fa-user-shield"></i>
                    </div>
                    <div class="sidebar-title"><?php echo BRAND_NAME; ?></div>
                    <div style="color:#e0e1dd;font-size:0.9em;">Admin Panel</div>
                </div>
                <div class="sidebar-section">Main</div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                <div class="sidebar-section">Customers</div>
                <a href="../customers.php"><i class="fas fa-users"></i> Customers</a>
                <a href="../cus-products.php"><i class="fas fa-users"></i> Customers Products</a>
                <div class="sidebar-section">Management</div>
                <a href="../pins/"><i class="fas fa-key"></i> PIN Management</a>
                <a href="../genealogy.php"><i class="fas fa-sitemap"></i> Genealogy</a>
                <a href="../reports/"><i class="fas fa-chart-bar"></i> Reports</a>
                <div class="sidebar-section">Support</div>
                <a href="index.php" class="active"><i class="fas fa-headset"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h2>Support Tickets</h2>
                    </div>
                    <div class="col-md-6 text-right">
                        <a href="create_ticket.php" class="btn btn-primary">
                            <i class="fas fa-plus mr-2"></i> Create Ticket
                        </a>
                    </div>
                </div>

                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stats-card bg-primary text-white">
                            <i class="fas fa-ticket-alt"></i>
                            <h3><?php echo safeNumberFormat($stats['total_tickets']); ?></h3>
                            <p class="mb-0">Total Tickets</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card bg-success text-white">
                            <i class="fas fa-plus-circle"></i>
                            <h3><?php echo safeNumberFormat($stats['new_tickets']); ?></h3>
                            <p class="mb-0">New Tickets</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card bg-info text-white">
                            <i class="fas fa-sync"></i>
                            <h3><?php echo safeNumberFormat($stats['in_process_tickets']); ?></h3>
                            <p class="mb-0">In Process</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card bg-secondary text-white">
                            <i class="fas fa-check-circle"></i>
                            <h3><?php echo safeNumberFormat($stats['closed_tickets']); ?></h3>
                            <p class="mb-0">Closed Tickets</p>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" action="" class="row">
                            <div class="col-md-4">
                                <div class="form-group mb-0">
                                    <label>Status</label>
                                    <select name="status" class="form-control" onchange="this.form.submit()">
                                        <option value="">All Status</option>
                                        <option value="created" <?php echo $status === 'created' ? 'selected' : ''; ?>>Created</option>
                                        <option value="active" <?php echo $status === 'active' ? 'selected' : ''; ?>>Active</option>
                                        <option value="in_process" <?php echo $status === 'in_process' ? 'selected' : ''; ?>>In Process</option>
                                        <option value="closed" <?php echo $status === 'closed' ? 'selected' : ''; ?>>Closed</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-0">
                                    <label>Search</label>
                                    <input type="text" name="search" class="form-control" value="<?php echo htmlspecialchars($search); ?>" 
                                           placeholder="Search by ticket ID, username, or subject...">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group mb-0">
                                    <label>&nbsp;</label>
                                    <button type="submit" class="btn btn-primary btn-block">
                                        <i class="fas fa-search mr-2"></i> Search
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Tickets Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Ticket ID</th>
                                        <th>User</th>
                                        <th>Subject</th>
                                        <th>Status</th>
                                        <th>Messages</th>
                                        <th>Created</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($tickets)): ?>
                                        <tr>
                                            <td colspan="7" class="text-center">No tickets found</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($tickets as $ticket): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($ticket['ticket_id']); ?></td>
                                                <td>
                                                    <?php echo htmlspecialchars($ticket['username']); ?><br>
                                                    <small class="text-muted"><?php echo htmlspecialchars($ticket['full_name']); ?></small>
                                                </td>
                                                <td><?php echo htmlspecialchars($ticket['subject']); ?></td>
                                                <td>
                                                    <span class="badge badge-<?php 
                                                        echo $ticket['status'] === 'created' ? 'primary' : 
                                                            ($ticket['status'] === 'active' ? 'success' : 
                                                            ($ticket['status'] === 'in_process' ? 'info' : 
                                                            ($ticket['status'] === 'closed' ? 'secondary' : 'warning'))); 
                                                    ?>">
                                                        <?php echo ucfirst(str_replace('_', ' ', $ticket['status'])); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="badge badge-pill badge-secondary">
                                                        <?php echo number_format($ticket['message_count']); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo $ticket['formatted_date']; ?></td>
                                                <td>
                                                    <a href="view_ticket.php?id=<?php echo $ticket['id']; ?>" class="btn btn-sm btn-info">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <?php if ($ticket['status'] !== 'closed'): ?>
                                                        <button class="btn btn-sm btn-success" onclick="updateStatus(<?php echo $ticket['id']; ?>, 'closed')">
                                                            <i class="fas fa-check"></i>
                                                        </button>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($totalPages > 1): ?>
                            <nav aria-label="Page navigation" class="mt-4">
                                <ul class="pagination justify-content-center">
                                    <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page - 1; ?>&status=<?php echo urlencode($status); ?>&search=<?php echo urlencode($search); ?>">Previous</a>
                                    </li>
                                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                                        <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                            <a class="page-link" href="?page=<?php echo $i; ?>&status=<?php echo urlencode($status); ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                                        </li>
                                    <?php endfor; ?>
                                    <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page + 1; ?>&status=<?php echo urlencode($status); ?>&search=<?php echo urlencode($search); ?>">Next</a>
                                    </li>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    $(function() {
        var sidebar = $('#adminSidebar');
        var overlay = $('#sidebarOverlay');
        sidebar.on('show.bs.collapse', function() {
            if (window.innerWidth < 768) overlay.show();
        });
        sidebar.on('hide.bs.collapse', function() {
            overlay.hide();
        });
        overlay.on('click', function() {
            sidebar.collapse('hide');
        });
        $(window).on('resize', function() {
            if (window.innerWidth >= 768) overlay.hide();
        });
    });
    function updateStatus(ticketId, status) {
        if (confirm('Are you sure you want to update this ticket\'s status?')) {
            $.ajax({
                url: 'update_status.php',
                method: 'POST',
                data: { ticket_id: ticketId, status: status },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        window.location.reload();
                    } else {
                        alert(response.message || 'Failed to update ticket status');
                    }
                },
                error: function() {
                    alert('An error occurred while updating the ticket status');
                }
            });
        }
    }
    </script>
</body>
</html> 