<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get POST data
$user_id = intval($_POST['user_id'] ?? 0);
$subject = trim($_POST['subject'] ?? '');
$message = trim($_POST['message'] ?? '');
$status = trim($_POST['status'] ?? 'created');

// Validate input
if (!$user_id || !$subject || !$message) {
    echo json_encode(['success' => false, 'message' => 'Missing required fields']);
    exit;
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Start transaction
    if (!$connection->begin_transaction()) {
        throw new Exception("Failed to start transaction");
    }

    // Generate ticket ID (Format: TID0225001)
    $query = "SELECT COUNT(*) as count FROM support_tickets";
    $result = $connection->query($query);
    $count = $result->fetch_assoc()['count'] + 1;
    $ticket_id = 'TID' . date('y') . str_pad($count, 5, '0', STR_PAD_LEFT);

    // Create ticket
    $query = "INSERT INTO support_tickets (ticket_id, user_id, subject, status) VALUES (?, ?, ?, ?)";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare ticket query: " . $connection->error);
    }
    
    $stmt->bind_param("siss", $ticket_id, $user_id, $subject, $status);
    if (!$stmt->execute()) {
        throw new Exception("Failed to create ticket: " . $stmt->error);
    }
    
    $ticket_db_id = $stmt->insert_id;

    // Add initial message
    $query = "INSERT INTO ticket_messages (ticket_id, sender_id, message) VALUES (?, ?, ?)";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare message query: " . $connection->error);
    }
    
    $admin_id = $user->getId();
    $stmt->bind_param("iis", $ticket_db_id, $admin_id, $message);
    if (!$stmt->execute()) {
        throw new Exception("Failed to add message: " . $stmt->error);
    }

    // Commit transaction
    if (!$connection->commit()) {
        throw new Exception("Failed to commit transaction");
    }

    echo json_encode([
        'success' => true,
        'message' => 'Ticket created successfully',
        'ticket_id' => $ticket_id
    ]);

} catch (Exception $e) {
    if (isset($connection) && $connection->ping()) {
        $connection->rollback();
    }
    
    error_log("Ticket Creation Error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while creating the ticket: ' . $e->getMessage()
    ]);
} 