<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get ticket ID from URL
$ticketId = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Get ticket details
$db = Database::getInstance();
$query = "SELECT st.*, u.username, u.full_name, u.email, u.phone 
          FROM support_tickets st 
          LEFT JOIN users u ON st.user_id = u.id 
          WHERE st.id = ?";
$stmt = $db->prepare($query);
$stmt->bind_param('i', $ticketId);
$stmt->execute();
$ticket = $stmt->get_result()->fetch_assoc();

if (!$ticket) {
    header('Location: tickets.php');
    exit;
}

// Get messages
$query = "SELECT tm.*, u.username, u.full_name, u.role 
          FROM ticket_messages tm 
          LEFT JOIN users u ON tm.sender_id = u.id 
          WHERE tm.ticket_id = ? 
          ORDER BY tm.created_at ASC";
$stmt = $db->prepare($query);
$stmt->bind_param('i', $ticketId);
$stmt->execute();
$messages = $stmt->get_result();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Support Chat</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
        .chat-container {
            height: calc(100vh - 300px);
            overflow-y: auto;
            background: #f8f9fa;
            padding: 20px;
            border-radius: 5px;
        }
        .message {
            margin-bottom: 20px;
            max-width: 80%;
        }
        .message.admin {
            margin-left: auto;
        }
        .message.user {
            margin-right: auto;
        }
        .message .content {
            padding: 10px 15px;
            border-radius: 10px;
            position: relative;
        }
        .message.admin .content {
            background: #007bff;
            color: white;
        }
        .message.user .content {
            background: #e9ecef;
        }
        .message .meta {
            font-size: 0.8em;
            margin-top: 5px;
            color: #6c757d;
        }
        .status-badge {
            width: 100px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="../customers.php"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="../pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="../reports/"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="tickets.php" class="active"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h2>Support Chat</h2>
                    </div>
                    <div class="col-md-6 text-right">
                        <a href="tickets.php" class="btn btn-secondary">
                            <i class="fas fa-arrow-left mr-2"></i>Back to Tickets
                        </a>
                    </div>
                </div>

                <!-- Ticket Info -->
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h5>Ticket Details</h5>
                                <p>
                                    <strong>Ticket ID:</strong> <?php echo htmlspecialchars($ticket['ticket_id']); ?><br>
                                    <strong>Subject:</strong> <?php echo htmlspecialchars($ticket['subject']); ?><br>
                                    <strong>Created:</strong> <?php echo date('M d, Y H:i', strtotime($ticket['created_at'])); ?>
                                </p>
                            </div>
                            <div class="col-md-6">
                                <h5>Customer Details</h5>
                                <p>
                                    <strong>Username:</strong> <?php echo htmlspecialchars($ticket['username']); ?><br>
                                    <strong>Name:</strong> <?php echo htmlspecialchars($ticket['full_name']); ?><br>
                                    <strong>Email:</strong> <?php echo htmlspecialchars($ticket['email']); ?><br>
                                    <strong>Phone:</strong> <?php echo htmlspecialchars($ticket['phone']); ?>
                                </p>
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-md-12">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <strong>Status:</strong>
                                        <?php
                                        $statusClass = [
                                            'created' => 'secondary',
                                            'active' => 'success',
                                            'in_process' => 'primary',
                                            'closed' => 'dark',
                                            'inactive' => 'warning',
                                            'deleted' => 'danger'
                                        ][$ticket['status']];
                                        ?>
                                        <span class="badge badge-<?php echo $statusClass; ?> status-badge">
                                            <?php echo ucfirst(str_replace('_', ' ', $ticket['status'])); ?>
                                        </span>
                                    </div>
                                    <button type="button" 
                                            class="btn btn-primary" 
                                            onclick="updateStatus(<?php echo $ticket['id']; ?>, '<?php echo $ticket['status']; ?>')">
                                        <i class="fas fa-sync-alt mr-2"></i>Update Status
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Chat Messages -->
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="chat-container" id="chatContainer">
                            <?php while ($message = $messages->fetch_assoc()): ?>
                            <div class="message <?php echo $message['role'] === 'admin' ? 'admin' : 'user'; ?>">
                                <div class="content">
                                    <?php echo nl2br(htmlspecialchars($message['message'])); ?>
                                </div>
                                <div class="meta">
                                    <?php echo htmlspecialchars($message['full_name']); ?> 
                                    (<?php echo htmlspecialchars($message['username']); ?>) - 
                                    <?php echo date('M d, Y H:i', strtotime($message['created_at'])); ?>
                                </div>
                            </div>
                            <?php endwhile; ?>
                        </div>

                        <!-- Message Input -->
                        <form id="replyForm" class="mt-4">
                            <input type="hidden" name="ticket_id" value="<?php echo $ticket['id']; ?>">
                            <div class="form-group">
                                <textarea class="form-control" name="message" rows="3" 
                                          placeholder="Type your reply here..." required></textarea>
                            </div>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-paper-plane mr-2"></i>Send Reply
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Status Update Modal -->
    <div class="modal fade" id="statusModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Update Ticket Status</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <form id="updateStatusForm">
                    <div class="modal-body">
                        <input type="hidden" id="ticketId" name="ticket_id">
                        <div class="form-group">
                            <label>New Status</label>
                            <select class="form-control" name="status" id="newStatus">
                                <option value="created">Created</option>
                                <option value="active">Active</option>
                                <option value="in_process">In Process</option>
                                <option value="closed">Closed</option>
                                <option value="inactive">Inactive</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Update Status</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Scroll to bottom of chat container
        function scrollToBottom() {
            const container = document.getElementById('chatContainer');
            container.scrollTop = container.scrollHeight;
        }

        // Call on page load
        scrollToBottom();

        function updateStatus(ticketId, currentStatus) {
            $('#ticketId').val(ticketId);
            $('#newStatus').val(currentStatus);
            $('#statusModal').modal('show');
        }

        $('#updateStatusForm').on('submit', function(e) {
            e.preventDefault();
            
            $.ajax({
                url: '../api/update_ticket_status.php',
                type: 'POST',
                data: $(this).serialize(),
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert(response.error || 'Failed to update ticket status');
                    }
                },
                error: function() {
                    alert('An error occurred while updating the ticket status');
                }
            });
        });

        $('#replyForm').on('submit', function(e) {
            e.preventDefault();
            
            $.ajax({
                url: '../api/add_ticket_reply.php',
                type: 'POST',
                data: $(this).serialize(),
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert(response.error || 'Failed to send reply');
                    }
                },
                error: function() {
                    alert('An error occurred while sending the reply');
                }
            });
        });
    </script>
</body>
</html> 