<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get filter parameters
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-d', strtotime('-30 days'));
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$interval = isset($_GET['interval']) ? $_GET['interval'] : 'daily';

$db = Database::getInstance();

// Get total users count
$totalUsersQuery = "SELECT COUNT(*) as total FROM users WHERE role = 'user'";
$totalUsers = $db->getConnection()->query($totalUsersQuery)->fetch_assoc()['total'];

// Get user growth data based on interval
switch ($interval) {
    case 'daily':
        $groupBy = "DATE(created_at)";
        $dateFormat = "%Y-%m-%d";
        break;
    case 'weekly':
        $groupBy = "YEARWEEK(created_at)";
        $dateFormat = "%Y-%u";
        break;
    case 'monthly':
        $groupBy = "DATE_FORMAT(created_at, '%Y-%m')";
        $dateFormat = "%Y-%m";
        break;
    default:
        $groupBy = "DATE(created_at)";
        $dateFormat = "%Y-%m-%d";
}

$growthQuery = "SELECT 
                    DATE_FORMAT(created_at, '$dateFormat') as period,
                    COUNT(*) as new_users,
                    SUM(COUNT(*)) OVER (ORDER BY MIN(created_at)) as cumulative_users
                FROM users 
                WHERE role = 'user' 
                    AND DATE(created_at) BETWEEN ? AND ?
                GROUP BY $groupBy
                ORDER BY MIN(created_at)";

$stmt = $db->getConnection()->prepare($growthQuery);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();
$growthData = $result->fetch_all(MYSQLI_ASSOC);

// Get status distribution
$statusQuery = "SELECT 
                    status,
                    COUNT(*) as count,
                    (COUNT(*) / (SELECT COUNT(*) FROM users WHERE role = 'user')) * 100 as percentage
                FROM users 
                WHERE role = 'user'
                GROUP BY status";
$statusData = $db->getConnection()->query($statusQuery)->fetch_all(MYSQLI_ASSOC);

// Prepare data for charts
$periods = [];
$newUsers = [];
$cumulativeUsers = [];
foreach ($growthData as $data) {
    $periods[] = $data['period'];
    $newUsers[] = $data['new_users'];
    $cumulativeUsers[] = $data['cumulative_users'];
}

$statusLabels = [];
$statusCounts = [];
$statusColors = [
    'active' => '#28a745',
    'inactive' => '#ffc107',
    'suspended' => '#dc3545'
];
foreach ($statusData as $status) {
    $statusLabels[] = ucfirst($status['status']);
    $statusCounts[] = $status['count'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - User Growth Report</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
        .chart-container {
            background-color: #fff;
            border-radius: 5px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="../customers.php"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="../pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="index.php"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="../support/"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>User Growth Report</h2>
                    <a href="index.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Reports
                    </a>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="get" class="row align-items-end">
                            <div class="col-md-3">
                                <label>Start Date</label>
                                <input type="text" class="form-control datepicker" name="start_date" value="<?php echo $start_date; ?>">
                            </div>
                            <div class="col-md-3">
                                <label>End Date</label>
                                <input type="text" class="form-control datepicker" name="end_date" value="<?php echo $end_date; ?>">
                            </div>
                            <div class="col-md-3">
                                <label>Interval</label>
                                <select name="interval" class="form-control">
                                    <option value="daily" <?php echo $interval === 'daily' ? 'selected' : ''; ?>>Daily</option>
                                    <option value="weekly" <?php echo $interval === 'weekly' ? 'selected' : ''; ?>>Weekly</option>
                                    <option value="monthly" <?php echo $interval === 'monthly' ? 'selected' : ''; ?>>Monthly</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-filter mr-2"></i> Apply Filters
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Summary Stats -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-body">
                                <h5 class="card-title">Total Users</h5>
                                <h2 class="mb-0"><?php echo number_format($totalUsers); ?></h2>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-body">
                                <h5 class="card-title">New Users (Selected Period)</h5>
                                <h2 class="mb-0"><?php echo number_format(array_sum($newUsers)); ?></h2>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-body">
                                <h5 class="card-title">Average Daily Growth</h5>
                                <h2 class="mb-0"><?php 
                                    $days = count($periods);
                                    echo $days > 0 ? number_format(array_sum($newUsers) / $days, 1) : '0';
                                ?></h2>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Growth Charts -->
                <div class="row">
                    <div class="col-md-8">
                        <div class="chart-container">
                            <h4>User Growth Trend</h4>
                            <canvas id="growthChart"></canvas>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="chart-container">
                            <h4>User Status Distribution</h4>
                            <canvas id="statusChart"></canvas>
                        </div>
                    </div>
                </div>

                <!-- Detailed Data -->
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0">Detailed Growth Data</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Period</th>
                                        <th>New Users</th>
                                        <th>Cumulative Users</th>
                                        <th>Growth Rate</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($growthData as $data): ?>
                                        <tr>
                                            <td><?php echo $data['period']; ?></td>
                                            <td><?php echo number_format($data['new_users']); ?></td>
                                            <td><?php echo number_format($data['cumulative_users']); ?></td>
                                            <td>
                                                <?php
                                                    $prevUsers = $data['cumulative_users'] - $data['new_users'];
                                                    $growthRate = $prevUsers > 0 ? 
                                                        (($data['new_users'] / $prevUsers) * 100) : 100;
                                                    echo number_format($growthRate, 1) . '%';
                                                ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
        // Initialize date pickers
        document.querySelectorAll('.datepicker').forEach(input => {
            flatpickr(input, {
                dateFormat: "Y-m-d"
            });
        });

        // Growth Chart
        const growthCtx = document.getElementById('growthChart').getContext('2d');
        new Chart(growthCtx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($periods); ?>,
                datasets: [{
                    label: 'New Users',
                    data: <?php echo json_encode($newUsers); ?>,
                    borderColor: '#007bff',
                    backgroundColor: 'rgba(0, 123, 255, 0.1)',
                    fill: true
                }, {
                    label: 'Cumulative Users',
                    data: <?php echo json_encode($cumulativeUsers); ?>,
                    borderColor: '#28a745',
                    backgroundColor: 'rgba(40, 167, 69, 0.1)',
                    fill: true
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Status Distribution Chart
        const statusCtx = document.getElementById('statusChart').getContext('2d');
        new Chart(statusCtx, {
            type: 'doughnut',
            data: {
                labels: <?php echo json_encode($statusLabels); ?>,
                datasets: [{
                    data: <?php echo json_encode($statusCounts); ?>,
                    backgroundColor: <?php echo json_encode(array_values($statusColors)); ?>
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    </script>
</body>
</html> 