<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get filter parameters
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-d', strtotime('-30 days'));
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$status = isset($_GET['status']) ? $_GET['status'] : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

$db = Database::getInstance();

// Get PIN statistics
$statsQuery = "SELECT 
                COUNT(*) as total_pins,
                SUM(CASE WHEN status = 'created' THEN 1 ELSE 0 END) as unused_pins,
                SUM(CASE WHEN status = 'activated' THEN 1 ELSE 0 END) as used_pins,
                SUM(CASE WHEN status = 'expired' THEN 1 ELSE 0 END) as expired_pins
              FROM pins";
$stats = $db->getConnection()->query($statsQuery)->fetch_assoc();

// Get daily PIN generation data
$dailyGenQuery = "SELECT 
                    DATE(created_at) as date,
                    COUNT(*) as pins_generated
                  FROM pins
                  WHERE DATE(created_at) BETWEEN ? AND ?
                  GROUP BY DATE(created_at)
                  ORDER BY date";
$stmt = $db->getConnection()->prepare($dailyGenQuery);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$dailyGenData = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get PIN usage data
$usageQuery = "SELECT 
                DATE(activated_at) as date,
                COUNT(*) as pins_used
               FROM pins 
               WHERE status = 'activated' 
                 AND DATE(activated_at) BETWEEN ? AND ?
               GROUP BY DATE(activated_at)
               ORDER BY date";
$stmt = $db->getConnection()->prepare($usageQuery);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$usageData = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Build conditions for detailed PIN list
$conditions = ["1=1"];
$params = [];
$types = "";

if ($status) {
    $conditions[] = "p.status = ?";
    $params[] = $status;
    $types .= "s";
}

if ($start_date) {
    $conditions[] = "DATE(p.created_at) >= ?";
    $params[] = $start_date;
    $types .= "s";
}

if ($end_date) {
    $conditions[] = "DATE(p.created_at) <= ?";
    $params[] = $end_date;
    $types .= "s";
}

// Get total records for pagination
$countQuery = "SELECT COUNT(*) as total FROM pins p WHERE " . implode(" AND ", $conditions);
$stmt = $db->getConnection()->prepare($countQuery);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$totalRecords = $stmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalRecords / $limit);

// Get detailed PIN list
$detailQuery = "SELECT p.*, 
                       u1.username as created_by_username,
                       u2.username as current_owner_username
                FROM pins p
                LEFT JOIN users u1 ON p.created_by = u1.id
                LEFT JOIN users u2 ON p.current_owner = u2.id
                WHERE " . implode(" AND ", $conditions) . "
                ORDER BY p.created_at DESC
                LIMIT ? OFFSET ?";
$types .= "ii";
$params[] = $limit;
$params[] = $offset;

$stmt = $db->getConnection()->prepare($detailQuery);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$pinDetails = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Prepare data for charts
$dates = [];
$pinsGenerated = [];
$pinsUsed = [];

foreach ($dailyGenData as $data) {
    $dates[] = $data['date'];
    $pinsGenerated[] = $data['pins_generated'];
}

$usageByDate = array_column($usageData, 'pins_used', 'date');
foreach ($dates as $date) {
    $pinsUsed[] = isset($usageByDate[$date]) ? $usageByDate[$date] : 0;
}

// Helper function to safely format numbers
function safeNumberFormat($value, $decimals = 0) {
    return number_format(floatval($value ?? 0), $decimals);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - PIN Usage Report</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
        .chart-container {
            background-color: #fff;
            border-radius: 5px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .status-badge {
            font-size: 0.875rem;
            padding: 5px 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="../customers.php"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="../pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="index.php"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="../support/"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>PIN Usage Report</h2>
                    <a href="index.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Reports
                    </a>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="get" class="row align-items-end">
                            <div class="col-md-3">
                                <label>Start Date</label>
                                <input type="text" class="form-control datepicker" name="start_date" value="<?php echo $start_date; ?>">
                            </div>
                            <div class="col-md-3">
                                <label>End Date</label>
                                <input type="text" class="form-control datepicker" name="end_date" value="<?php echo $end_date; ?>">
                            </div>
                            <div class="col-md-3">
                                <label>Status</label>
                                <select name="status" class="form-control">
                                    <option value="">All</option>
                                    <option value="created" <?php echo $status === 'created' ? 'selected' : ''; ?>>Unused</option>
                                    <option value="activated" <?php echo $status === 'activated' ? 'selected' : ''; ?>>Used</option>
                                    <option value="expired" <?php echo $status === 'expired' ? 'selected' : ''; ?>>Expired</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-filter mr-2"></i> Apply Filters
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Summary Stats -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card bg-primary text-white">
                            <div class="card-body">
                                <h5 class="card-title">Total PINs</h5>
                                <h2 class="mb-0"><?php echo safeNumberFormat($stats['total_pins']); ?></h2>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-success text-white">
                            <div class="card-body">
                                <h5 class="card-title">Unused PINs</h5>
                                <h2 class="mb-0"><?php echo safeNumberFormat($stats['unused_pins']); ?></h2>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-info text-white">
                            <div class="card-body">
                                <h5 class="card-title">Used PINs</h5>
                                <h2 class="mb-0"><?php echo safeNumberFormat($stats['used_pins']); ?></h2>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-danger text-white">
                            <div class="card-body">
                                <h5 class="card-title">Expired PINs</h5>
                                <h2 class="mb-0"><?php echo safeNumberFormat($stats['expired_pins']); ?></h2>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Charts -->
                <div class="row mb-4">
                    <div class="col-md-12">
                        <div class="chart-container">
                            <canvas id="pinUsageChart"></canvas>
                        </div>
                        </div>
                    </div>
                </div>

                <!-- PIN Details Table -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">PIN Details</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>PIN Number</th>
                                        <th>Status</th>
                                        <th>Created By</th>
                                        <th>Current Owner</th>
                                        <th>Created At</th>
                                        <th>Activated At</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($pinDetails)): ?>
                                        <tr>
                                            <td colspan="6" class="text-center">No PINs found</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($pinDetails as $pin): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($pin['pin_number']); ?></td>
                                                <td>
                                                    <span class="badge status-badge <?php 
                                                        echo $pin['status'] === 'created' ? 'badge-success' : 
                                                            ($pin['status'] === 'activated' ? 'badge-info' : 'badge-danger'); 
                                                    ?>">
                                                        <?php echo ucfirst($pin['status']); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo htmlspecialchars($pin['created_by_username']); ?></td>
                                                <td><?php echo htmlspecialchars($pin['current_owner_username']); ?></td>
                                                <td><?php echo date('M d, Y H:i', strtotime($pin['created_at'])); ?></td>
                                                <td><?php echo $pin['activated_at'] ? date('M d, Y H:i', strtotime($pin['activated_at'])) : '-'; ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($totalPages > 1): ?>
                            <nav aria-label="Page navigation" class="mt-4">
                                <ul class="pagination justify-content-center">
                                    <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page - 1; ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&status=<?php echo urlencode($status); ?>">Previous</a>
                                    </li>
                                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                                        <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                            <a class="page-link" href="?page=<?php echo $i; ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&status=<?php echo urlencode($status); ?>"><?php echo $i; ?></a>
                                        </li>
                                    <?php endfor; ?>
                                    <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page + 1; ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&status=<?php echo urlencode($status); ?>">Next</a>
                                    </li>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
        // Initialize date pickers
        flatpickr(".datepicker", {
            dateFormat: "Y-m-d"
        });

        // Initialize chart
        const ctx = document.getElementById('pinUsageChart').getContext('2d');
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($dates); ?>,
                datasets: [{
                    label: 'PINs Generated',
                    data: <?php echo json_encode($pinsGenerated); ?>,
                    borderColor: 'rgb(75, 192, 192)',
                    tension: 0.1
                }, {
                    label: 'PINs Used',
                    data: <?php echo json_encode($pinsUsed); ?>,
                    borderColor: 'rgb(255, 99, 132)',
                    tension: 0.1
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                }
            }
        });
    </script>
</body>
</html> 