<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get filter parameters
$report_type = $_GET['type'] ?? 'customers';
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');
$search = trim($_GET['search'] ?? '');
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 10;
$offset = ($page - 1) * $limit;

// Initialize database connection
$db = Database::getInstance();
$connection = $db->getConnection();

// Function to safely format numbers
function safeNumberFormat($number, $decimals = 2) {
    return number_format(floatval($number ?? 0), $decimals);
}

// Get statistics
$stats = [
    'total_users' => 0,
    'active_users' => 0,
    'total_earnings' => 0,
    'pending_withdrawals' => 0
];

// Get total users
$query = "SELECT 
            COALESCE(COUNT(*), 0) as total_users,
            COALESCE(SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END), 0) as active_users
          FROM users WHERE role = 'user'";
$result = $connection->query($query);
if ($result) {
    $row = $result->fetch_assoc();
    $stats['total_users'] = $row['total_users'];
    $stats['active_users'] = $row['active_users'];
}

// Get total earnings
$query = "SELECT COALESCE(SUM(amount), 0) as total FROM earnings";
$result = $connection->query($query);
if ($result) {
    $stats['total_earnings'] = $result->fetch_assoc()['total'];
}

// Get pending withdrawals
$query = "SELECT COALESCE(SUM(amount), 0) as total FROM withdrawals WHERE status = 'pending'";
$result = $connection->query($query);
if ($result) {
    $stats['pending_withdrawals'] = $result->fetch_assoc()['total'];
}

// Build query based on report type
switch ($report_type) {
    case 'earnings':
        $baseQuery = "FROM earnings e 
                     JOIN users u ON e.user_id = u.id 
                     WHERE e.created_at BETWEEN ? AND ?";
        if ($search) {
            $baseQuery .= " AND (u.username LIKE ? OR u.full_name LIKE ?)";
            $params = [$start_date, $end_date, "%$search%", "%$search%"];
            $types = "ssss";
        } else {
            $params = [$start_date, $end_date];
            $types = "ss";
        }
        break;

    case 'withdrawals':
        $baseQuery = "FROM withdrawals w 
                     JOIN users u ON w.user_id = u.id 
                     WHERE w.created_at BETWEEN ? AND ?";
        if ($search) {
            $baseQuery .= " AND (u.username LIKE ? OR u.full_name LIKE ?)";
            $params = [$start_date, $end_date, "%$search%", "%$search%"];
            $types = "ssss";
        } else {
            $params = [$start_date, $end_date];
            $types = "ss";
        }
        break;

    default: // customers
        $baseQuery = "FROM users u 
                     LEFT JOIN user_profiles up ON u.id = up.user_id 
                     WHERE u.role = 'user'";
        if ($search) {
            $baseQuery .= " AND (u.username LIKE ? OR u.full_name LIKE ? OR u.email LIKE ?)";
            $params = ["%$search%", "%$search%", "%$search%"];
            $types = "sss";
        } else {
            $params = [];
            $types = "";
        }
}

// Get total records for pagination
$countQuery = "SELECT COUNT(*) as total " . $baseQuery;
$stmt = $connection->prepare($countQuery);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$totalRecords = $stmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalRecords / $limit);

// Get data with pagination
switch ($report_type) {
    case 'earnings':
        $dataQuery = "SELECT e.*, u.username, u.full_name, 
                            DATE_FORMAT(e.created_at, '%Y-%m-%d %H:%i') as formatted_date 
                     " . $baseQuery . " 
                     ORDER BY e.created_at DESC 
                     LIMIT ? OFFSET ?";
        break;

    case 'withdrawals':
        $dataQuery = "SELECT w.*, u.username, u.full_name, 
                            DATE_FORMAT(w.created_at, '%Y-%m-%d %H:%i') as formatted_date 
                     " . $baseQuery . " 
                     ORDER BY w.created_at DESC 
                     LIMIT ? OFFSET ?";
        break;

    default: // customers
        $dataQuery = "SELECT u.*, up.kyc_status, 
                            DATE_FORMAT(u.created_at, '%Y-%m-%d %H:%i') as formatted_date 
                     " . $baseQuery . " 
                     ORDER BY u.created_at DESC 
                     LIMIT ? OFFSET ?";
}

$params[] = $limit;
$params[] = $offset;
$types .= "ii";

$stmt = $connection->prepare($dataQuery);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$data = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Reports</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(180deg, #3a86ff 0%, #8338ec 100%);
            padding-top: 20px;
            border-top-right-radius: 20px;
            border-bottom-right-radius: 20px;
            box-shadow: 2px 0 12px rgba(51,51,51,0.08);
            transition: background 0.3s;
        }
        .sidebar .sidebar-brand {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 2rem;
        }
        .sidebar .sidebar-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: #fff;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.5rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .sidebar .sidebar-logo i {
            font-size: 2rem;
            color: #3a86ff;
        }
        .sidebar .sidebar-title {
            color: #fff;
            font-weight: bold;
            font-size: 1.2rem;
            letter-spacing: 1px;
        }
        .sidebar-section {
            color: #e0e1dd;
            font-size: 0.85rem;
            text-transform: uppercase;
            margin: 1.5rem 0 0.5rem 1rem;
            letter-spacing: 0.08em;
        }
        .sidebar a {
            color: #fff;
            padding: 12px 22px;
            display: flex;
            align-items: center;
            border-radius: 8px 0 0 8px;
            margin-bottom: 6px;
            font-size: 1.05rem;
            transition: background 0.2s, color 0.2s, box-shadow 0.2s;
        }
        .sidebar a.active, .sidebar a:hover {
            background: rgba(255, 255, 255, 0.15);
            color: #ffbe0b;
            box-shadow: 2px 2px 8px rgba(0,0,0,0.07);
            font-weight: bold;
        }
        .sidebar a i {
            margin-right: 12px;
            font-size: 1.2em;
        }
        @media (max-width: 767.98px) {
            .sidebar {
                position: fixed;
                z-index: 1050;
                left: 0;
                top: 0;
                width: 220px;
                height: 100vh;
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                border-radius: 0 20px 20px 0;
            }
            .sidebar.collapse.show {
                transform: translateX(0);
            }
            .main-content {
                padding: 10px;
            }
        }
        .main-content {
            padding: 20px;
        }
        .stats-card {
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .stats-card i {
            font-size: 2rem;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar Overlay (for mobile) -->
            <div id="sidebarOverlay" style="display:none;position:fixed;top:0;left:0;width:100vw;height:100vh;background:rgba(0,0,0,0.3);z-index:1049;"></div>
            <!-- Sidebar Toggle Button (visible on mobile) -->
            <button class="btn btn-dark d-md-none m-2" type="button" data-toggle="collapse" data-target="#adminSidebar" aria-controls="adminSidebar" aria-expanded="false" aria-label="Toggle navigation">
                <i class="fas fa-bars"></i> Menu
            </button>
            <!-- Sidebar -->
            <nav id="adminSidebar" class="col-md-2 sidebar collapse d-md-block">
                <div class="sidebar-brand">
                    <div class="sidebar-logo">
                        <i class="fas fa-user-shield"></i>
                    </div>
                    <div class="sidebar-title"><?php echo BRAND_NAME; ?></div>
                    <div style="color:#e0e1dd;font-size:0.9em;">Admin Panel</div>
                </div>
                <div class="sidebar-section">Main</div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                <div class="sidebar-section">Customers</div>
                <a href="../customers.php"><i class="fas fa-users"></i> Customers</a>
                <a href="../cus-products.php"><i class="fas fa-users"></i> Customers Products</a>
                <div class="sidebar-section">Management</div>
                <a href="../pins/"><i class="fas fa-key"></i> PIN Management</a>
                <a href="../genealogy.php"><i class="fas fa-sitemap"></i> Genealogy</a>
                <a href="index.php" class="active"><i class="fas fa-chart-bar"></i> Reports</a>
                <div class="sidebar-section">Support</div>
                <a href="../support/"><i class="fas fa-headset"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h2>Reports</h2>
                    </div>
                    <div class="col-md-6 text-right">
                        <button class="btn btn-primary" onclick="exportReport()">
                            <i class="fas fa-download mr-2"></i> Export Report
                        </button>
                    </div>
                </div>

                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stats-card bg-primary text-white">
                            <i class="fas fa-users"></i>
                            <h3><?php echo number_format($stats['total_users']); ?></h3>
                            <p class="mb-0">Total Users</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card bg-success text-white">
                            <i class="fas fa-user-check"></i>
                            <h3><?php echo number_format($stats['active_users']); ?></h3>
                            <p class="mb-0">Active Users</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card bg-info text-white">
                            <i class="fas fa-rupee-sign"></i>
                            <h3><?php echo safeNumberFormat($stats['total_earnings']); ?></h3>
                            <p class="mb-0">Total Earnings</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card bg-warning text-white">
                            <i class="fas fa-clock"></i>
                            <h3><?php echo safeNumberFormat($stats['pending_withdrawals']); ?></h3>
                            <p class="mb-0">Pending Withdrawals</p>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" action="" class="row">
                            <div class="col-md-3">
                                <div class="form-group mb-0">
                                    <label>Report Type</label>
                                    <select name="type" class="form-control" onchange="this.form.submit()">
                                        <option value="customers" <?php echo $report_type === 'customers' ? 'selected' : ''; ?>>Customer List</option>
                                        <option value="earnings" <?php echo $report_type === 'earnings' ? 'selected' : ''; ?>>Earnings List</option>
                                        <option value="withdrawals" <?php echo $report_type === 'withdrawals' ? 'selected' : ''; ?>>Withdrawal Report</option>
                                    </select>
                                </div>
                            </div>
                            <?php if ($report_type !== 'customers'): ?>
                            <div class="col-md-3">
                                <div class="form-group mb-0">
                                    <label>Date Range</label>
                                    <input type="text" name="daterange" class="form-control" value="<?php echo $start_date . ' - ' . $end_date; ?>">
                                    <input type="hidden" name="start_date" value="<?php echo $start_date; ?>">
                                    <input type="hidden" name="end_date" value="<?php echo $end_date; ?>">
                                </div>
                            </div>
                            <?php endif; ?>
                            <div class="col-md-3">
                                <div class="form-group mb-0">
                                    <label>Search</label>
                                    <input type="text" name="search" class="form-control" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search...">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group mb-0">
                                    <label>&nbsp;</label>
                                    <button type="submit" class="btn btn-primary btn-block">
                                        <i class="fas fa-search mr-2"></i> Search
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-1">
                                <div class="form-group mb-0">
                                    <label>&nbsp;</label>
                                    <a href="index.php" class="btn btn-secondary btn-block">
                                        <i class="fas fa-redo"></i>
                                    </a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Data Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <?php if ($report_type === 'customers'): ?>
                                            <th>Username</th>
                                            <th>Full Name</th>
                                            <th>Email</th>
                                            <th>Status</th>
                                            <th>KYC Status</th>
                                            <th>Registered On</th>
                                        <?php elseif ($report_type === 'earnings'): ?>
                                            <th>Username</th>
                                            <th>Full Name</th>
                                            <th>Amount</th>
                                            <th>Type</th>
                                            <th>Status</th>
                                            <th>Date</th>
                                        <?php else: ?>
                                            <th>Username</th>
                                            <th>Full Name</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                            <th>Requested On</th>
                                            <th>Processed On</th>
                                        <?php endif; ?>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($data)): ?>
                                        <tr>
                                            <td colspan="6" class="text-center">No records found</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($data as $row): ?>
                                            <tr id="row-<?php echo $row['id']; ?>">
                                                <?php if ($report_type === 'customers'): ?>
                                                    <td><?php echo htmlspecialchars($row['username']); ?></td>
                                                    <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                                                    <td><?php echo htmlspecialchars($row['email']); ?></td>
                                                    <td>
                                                        <span class="badge badge-<?php echo $row['status'] === 'active' ? 'success' : 'danger'; ?>">
                                                            <?php echo ucfirst($row['status']); ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <span class="badge badge-<?php 
                                                            echo $row['kyc_status'] === 'verified' ? 'success' : 
                                                                ($row['kyc_status'] === 'rejected' ? 'danger' : 'warning'); 
                                                        ?>">
                                                            <?php echo ucfirst($row['kyc_status'] ?? 'pending'); ?>
                                                        </span>
                                                    </td>
                                                    <td><?php echo $row['formatted_date']; ?></td>
                                                <?php elseif ($report_type === 'earnings'): ?>
                                                    <td><?php echo htmlspecialchars($row['username']); ?></td>
                                                    <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                                                    <td>₹<?php echo safeNumberFormat($row['amount']); ?></td>
                                                    <td><?php echo ucfirst($row['earning_type']); ?></td>
                                                    <td>
                                                        <span class="badge badge-<?php 
                                                            echo $row['status'] === 'available' ? 'success' : 
                                                                ($row['status'] === 'withdrawn' ? 'info' : 'warning'); 
                                                        ?>">
                                                            <?php echo ucfirst($row['status']); ?>
                                                        </span>
                                                    </td>
                                                    <td><?php echo $row['formatted_date']; ?></td>
                                                <?php else: ?>
                                                    <td><?php echo htmlspecialchars($row['username']); ?></td>
                                                    <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                                                    <td>₹<?php echo safeNumberFormat($row['amount']); ?></td>
                                                    <td class="status-cell">
                                                        <span class="badge badge-<?php 
                                                            echo $row['status'] === 'pending' ? 'warning' : ($row['status'] === 'approved' ? 'success' : 'danger'); 
                                                        ?>"><?php echo ucfirst($row['status']); ?></span>
                                                    </td>
                                                    <td><?php echo $row['formatted_date']; ?></td>
                                                    <td><?php echo $row['processed_at'] ? date('Y-m-d H:i', strtotime($row['processed_at'])) : '-'; ?></td>
                                                    <td class="actions-cell">
                                                        <?php if ($row['status'] === 'pending'): ?>
                                                        <button class="btn btn-success btn-sm" onclick="updateWithdrawal(<?php echo $row['id']; ?>, 'approved')">Approve</button>
                                                        <button class="btn btn-danger btn-sm" onclick="updateWithdrawal(<?php echo $row['id']; ?>, 'rejected')">Reject</button>
                                                        <?php else: ?>
                                                        -
                                                        <?php endif; ?>
                                                    </td>
                                                <?php endif; ?>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($totalPages > 1): ?>
                            <nav aria-label="Page navigation" class="mt-4">
                                <ul class="pagination justify-content-center">
                                    <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page - 1; ?>&type=<?php echo $report_type; ?>&search=<?php echo urlencode($search); ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>">Previous</a>
                                    </li>
                                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                                        <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                            <a class="page-link" href="?page=<?php echo $i; ?>&type=<?php echo $report_type; ?>&search=<?php echo urlencode($search); ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>"><?php echo $i; ?></a>
                                        </li>
                                    <?php endfor; ?>
                                    <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page + 1; ?>&type=<?php echo $report_type; ?>&search=<?php echo urlencode($search); ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>">Next</a>
                                    </li>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>
    <script>
        $(document).ready(function() {
            $('input[name="daterange"]').daterangepicker({
                startDate: moment('<?php echo $start_date; ?>'),
                endDate: moment('<?php echo $end_date; ?>'),
                locale: {
                    format: 'YYYY-MM-DD'
                }
            }, function(start, end) {
                $('input[name="start_date"]').val(start.format('YYYY-MM-DD'));
                $('input[name="end_date"]').val(end.format('YYYY-MM-DD'));
            });
        });

        function exportReport() {
            const type = '<?php echo $report_type; ?>';
            const search = '<?php echo urlencode($search); ?>';
            const startDate = '<?php echo $start_date; ?>';
            const endDate = '<?php echo $end_date; ?>';
            
            window.location.href = `export.php?type=${type}&search=${search}&start_date=${startDate}&end_date=${endDate}`;
        }

        $(function() {
            var sidebar = $('#adminSidebar');
            var overlay = $('#sidebarOverlay');
            sidebar.on('show.bs.collapse', function() {
                if (window.innerWidth < 768) overlay.show();
            });
            sidebar.on('hide.bs.collapse', function() {
                overlay.hide();
            });
            overlay.on('click', function() {
                sidebar.collapse('hide');
            });
            $(window).on('resize', function() {
                if (window.innerWidth >= 768) overlay.hide();
            });
        });
    </script>
    <?php if ($report_type === 'withdrawals'): ?>
    <script>
    function updateWithdrawal(id, status) {
        if (!confirm('Are you sure you want to ' + status + ' this withdrawal?')) return;
        fetch('../withdrawals/update.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: 'id=' + encodeURIComponent(id) + '&status=' + encodeURIComponent(status)
        })
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const row = document.getElementById('row-' + id);
                if (row) {
                    // Update status badge
                    const statusCell = row.querySelector('td.status-cell');
                    if (statusCell) {
                        statusCell.innerHTML = `<span class="badge badge-${status === 'approved' ? 'success' : 'danger'}">${status.charAt(0).toUpperCase() + status.slice(1)}</span>`;
                    }
                    // Remove actions
                    const actionsCell = row.querySelector('td.actions-cell');
                    if (actionsCell) actionsCell.innerHTML = '-';
                }
                alert('Withdrawal ' + status + ' successfully!');
            } else {
                alert(data.message || 'Failed to update withdrawal');
            }
        })
        .catch(() => alert('Error updating withdrawal'));
    }
    </script>
    <?php endif; ?>
</body>
</html> 