<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get filter parameters
$start_date = $_GET['start_date'] ?? date('Y-m-01'); // First day of current month
$end_date = $_GET['end_date'] ?? date('Y-m-d'); // Today
$status = $_GET['status'] ?? '';
$user_id = intval($_GET['user_id'] ?? 0);
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 10;
$offset = ($page - 1) * $limit;

// Build the base query
$db = Database::getInstance();
$baseQuery = "FROM earnings e 
              LEFT JOIN users u ON e.user_id = u.id 
              WHERE e.created_at BETWEEN ? AND ?";
$params = [$start_date . ' 00:00:00', $end_date . ' 23:59:59'];
$types = "ss";

// Add status condition if provided
if ($status) {
    $baseQuery .= " AND e.status = ?";
    $params[] = $status;
    $types .= "s";
}

// Add user condition if provided
if ($user_id) {
    $baseQuery .= " AND e.user_id = ?";
    $params[] = $user_id;
    $types .= "i";
}

// Get total records for pagination
$countQuery = "SELECT COUNT(*) as total " . $baseQuery;
$stmt = $db->getConnection()->prepare($countQuery);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$totalRecords = $stmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalRecords / $limit);

// Get data with pagination
$dataQuery = "SELECT e.*, u.username, u.full_name " . $baseQuery . " 
              ORDER BY e.created_at DESC 
              LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= "ii";

$stmt = $db->getConnection()->prepare($dataQuery);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$earnings = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get summary statistics
$statsQuery = "SELECT 
                COUNT(*) as total_transactions,
                SUM(amount) as total_amount,
                SUM(CASE WHEN status = 'available' THEN amount ELSE 0 END) as available_amount,
                SUM(CASE WHEN status = 'withdrawn' THEN amount ELSE 0 END) as withdrawn_amount
              FROM earnings e 
              WHERE e.created_at BETWEEN ? AND ?";
$params = [$start_date . ' 00:00:00', $end_date . ' 23:59:59'];
$types = "ss";

if ($user_id) {
    $statsQuery .= " AND e.user_id = ?";
    $params[] = $user_id;
    $types .= "i";
}

$stmt = $db->getConnection()->prepare($statsQuery);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$stats = $stmt->get_result()->fetch_assoc();

// Get user details if user_id is provided
$customer = null;
if ($user_id) {
    $userQuery = "SELECT username, full_name FROM users WHERE id = ?";
    $stmt = $db->getConnection()->prepare($userQuery);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $customer = $stmt->get_result()->fetch_assoc();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Earnings Report</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
        .stats-card {
            background-color: #f8f9fa;
            border-radius: 5px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .status-badge {
            font-size: 0.875rem;
            padding: 5px 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="../customers.php"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="../pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="index.php" class="active"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="../support/"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>
                        Earnings Report
                        <?php if ($customer): ?>
                            <small class="text-muted">for <?php echo htmlspecialchars($customer['full_name']); ?></small>
                        <?php endif; ?>
                    </h2>
                    <div>
                        <?php if ($customer): ?>
                            <a href="../customer_view.php?id=<?php echo $user_id; ?>" class="btn btn-secondary">
                                <i class="fas fa-arrow-left"></i> Back to Customer
                            </a>
                        <?php else: ?>
                            <a href="index.php" class="btn btn-secondary">
                                <i class="fas fa-arrow-left"></i> Back to Reports
                            </a>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Statistics Cards -->
                <div class="row">
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h5>Total Transactions</h5>
                            <h3><?php echo number_format($stats['total_transactions'] ?? 0); ?></h3>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h5>Total Amount</h5>
                            <h3>₹<?php echo number_format($stats['total_amount'] ?? 0, 2); ?></h3>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h5>Available Amount</h5>
                            <h3>₹<?php echo number_format($stats['available_amount'] ?? 0, 2); ?></h3>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h5>Withdrawn Amount</h5>
                            <h3>₹<?php echo number_format($stats['withdrawn_amount'] ?? 0, 2); ?></h3>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" action="" class="row">
                            <?php if ($user_id): ?>
                                <input type="hidden" name="user_id" value="<?php echo $user_id; ?>">
                            <?php endif; ?>
                            <div class="col-md-3">
                                <div class="form-group mb-0">
                                    <label>Start Date</label>
                                    <input type="text" name="start_date" class="form-control datepicker" value="<?php echo $start_date; ?>" required>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group mb-0">
                                    <label>End Date</label>
                                    <input type="text" name="end_date" class="form-control datepicker" value="<?php echo $end_date; ?>" required>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group mb-0">
                                    <label>Status</label>
                                    <select name="status" class="form-control">
                                        <option value="">All Status</option>
                                        <option value="available" <?php echo $status === 'available' ? 'selected' : ''; ?>>Available</option>
                                        <option value="withdrawn" <?php echo $status === 'withdrawn' ? 'selected' : ''; ?>>Withdrawn</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group mb-0">
                                    <label>&nbsp;</label>
                                    <button type="submit" class="btn btn-primary btn-block">
                                        <i class="fas fa-filter mr-2"></i> Filter
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group mb-0">
                                    <label>&nbsp;</label>
                                    <a href="?<?php echo $user_id ? 'user_id=' . $user_id : ''; ?>" class="btn btn-secondary btn-block">
                                        <i class="fas fa-redo mr-2"></i> Reset
                                    </a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Earnings Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Transaction ID</th>
                                        <?php if (!$user_id): ?>
                                            <th>User</th>
                                        <?php endif; ?>
                                        <th>Amount</th>
                                        <th>Type</th>
                                        <th>Status</th>
                                        <th>Description</th>
                                        <th>Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($earnings)): ?>
                                        <tr>
                                            <td colspan="<?php echo $user_id ? '6' : '7'; ?>" class="text-center">No earnings found</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($earnings as $earning): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($earning['transaction_id'] ?? '-'); ?></td>
                                                <?php if (!$user_id): ?>
                                                    <td>
                                                        <a href="../customer_view.php?id=<?php echo $earning['user_id']; ?>">
                                                            <?php echo htmlspecialchars($earning['username']); ?>
                                                            <br>
                                                            <small class="text-muted"><?php echo htmlspecialchars($earning['full_name']); ?></small>
                                                        </a>
                                                    </td>
                                                <?php endif; ?>
                                                <td>₹<?php echo number_format($earning['amount'], 2); ?></td>
                                                <td><?php echo ucfirst($earning['type'] ?? 'cash'); ?></td>
                                                <td>
                                                    <span class="badge status-badge <?php 
                                                        echo ($earning['status'] ?? '') === 'available' ? 'badge-success' : 'badge-primary'; 
                                                    ?>">
                                                        <?php echo ucfirst($earning['status'] ?? 'pending'); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo htmlspecialchars($earning['description'] ?? '-'); ?></td>
                                                <td><?php echo date('M d, Y H:i', strtotime($earning['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($totalPages > 1): ?>
                            <nav aria-label="Page navigation" class="mt-4">
                                <ul class="pagination justify-content-center">
                                    <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page - 1; ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&status=<?php echo urlencode($status); ?><?php echo $user_id ? '&user_id=' . $user_id : ''; ?>">
                                            Previous
                                        </a>
                                    </li>
                                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                                        <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                            <a class="page-link" href="?page=<?php echo $i; ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&status=<?php echo urlencode($status); ?><?php echo $user_id ? '&user_id=' . $user_id : ''; ?>">
                                                <?php echo $i; ?>
                                            </a>
                                        </li>
                                    <?php endfor; ?>
                                    <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page + 1; ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&status=<?php echo urlencode($status); ?><?php echo $user_id ? '&user_id=' . $user_id : ''; ?>">
                                            Next
                                        </a>
                                    </li>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    <script>
        // Initialize date pickers
        flatpickr(".datepicker", {
            dateFormat: "Y-m-d",
            maxDate: "today"
        });
    </script>
</body>
</html> 