<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get POST data
$pins = isset($_POST['pins']) ? json_decode($_POST['pins'], true) : [];
$pinCount = isset($_POST['count']) ? (int)$_POST['count'] : 0;
$to_username = trim($_POST['username'] ?? '');

// Validate input
if (empty($to_username)) {
    echo json_encode(['success' => false, 'message' => 'Username is required']);
    exit;
}

if (empty($pins) && $pinCount <= 0) {
    echo json_encode(['success' => false, 'message' => 'No PINs selected for transfer']);
    exit;
}

// Prevent transfer to self
if ($to_username === $_SESSION['username']) {
    echo json_encode(['success' => false, 'message' => 'Cannot transfer PINs to yourself']);
    exit;
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Start transaction
    if (!$connection->begin_transaction()) {
        throw new Exception("Failed to start transaction");
    }

    // Get target user ID
    $userQuery = "SELECT id FROM users WHERE username = ? AND status = 'active'";
    $stmt = $connection->prepare($userQuery);
    if (!$stmt) {
        throw new Exception("Failed to prepare user query: " . $connection->error);
    }
    
    $stmt->bind_param("s", $to_username);
    if (!$stmt->execute()) {
        throw new Exception("Failed to execute user query: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
    if ($result->num_rows === 0) {
        throw new Exception("Invalid or inactive username");
    }
    
    $to_user_id = $result->fetch_assoc()['id'];
    $from_user_id = $user->getId();

    // Get pins by count if needed
    if ($pinCount > 0 && empty($pins)) {
        $pinsQuery = "SELECT pin_number FROM pins 
                     WHERE current_owner = ? AND status = 'created' 
                     ORDER BY RAND() LIMIT ?";
        $pinsStmt = $connection->prepare($pinsQuery);
        if (!$pinsStmt) {
            throw new Exception("Failed to prepare pins query: " . $connection->error);
        }
        
        $pinsStmt->bind_param("ii", $from_user_id, $pinCount);
        if (!$pinsStmt->execute()) {
            throw new Exception("Failed to execute pins query: " . $pinsStmt->error);
        }
        
        $pinsResult = $pinsStmt->get_result();
        $pins = [];
        
        while ($row = $pinsResult->fetch_assoc()) {
            $pins[] = $row['pin_number'];
        }
        
        if (count($pins) < $pinCount) {
            throw new Exception("Only " . count($pins) . " PINs are available for transfer");
        }
    }
    
    if (empty($pins)) {
        throw new Exception("No valid PINs to transfer");
    }

    // Update PINs
    $updateQuery = "UPDATE pins SET current_owner = ?, status = 'transferred' WHERE pin_number = ? AND current_owner = ? AND status = 'created'";
    $stmt = $connection->prepare($updateQuery);
    if (!$stmt) {
        throw new Exception("Failed to prepare update query: " . $connection->error);
    }

    // Record transfers
    $transferQuery = "INSERT INTO pin_transfers (pin_id, from_user, to_user) VALUES (?, ?, ?)";
    $transferStmt = $connection->prepare($transferQuery);
    if (!$transferStmt) {
        throw new Exception("Failed to prepare transfer query: " . $connection->error);
    }

    $transferred_pins = [];
    foreach ($pins as $pin) {
        // Get PIN ID and verify ownership
        $pinQuery = "SELECT id FROM pins WHERE pin_number = ? AND current_owner = ? AND status = 'created'";
        $pinStmt = $connection->prepare($pinQuery);
        if (!$pinStmt) {
            throw new Exception("Failed to prepare PIN query: " . $connection->error);
        }
        
        $pinStmt->bind_param("si", $pin, $from_user_id);
        if (!$pinStmt->execute()) {
            throw new Exception("Failed to execute PIN query: " . $pinStmt->error);
        }
        
        $pinResult = $pinStmt->get_result();
        if ($pinResult->num_rows === 0) {
            continue; // Skip if PIN not found or not available
        }
        
        $pin_id = $pinResult->fetch_assoc()['id'];

        // Update PIN ownership
        $stmt->bind_param("isi", $to_user_id, $pin, $from_user_id);
        if (!$stmt->execute()) {
            throw new Exception("Failed to update PIN ownership: " . $stmt->error);
        }

        // Record transfer
        $transferStmt->bind_param("iii", $pin_id, $from_user_id, $to_user_id);
        if (!$transferStmt->execute()) {
            throw new Exception("Failed to record transfer: " . $transferStmt->error);
        }

        $transferred_pins[] = $pin;
    }

    if (empty($transferred_pins)) {
        throw new Exception("No valid PINs to transfer");
    }

    // Commit transaction
    if (!$connection->commit()) {
        throw new Exception("Failed to commit transaction");
    }

    echo json_encode([
        'success' => true,
        'message' => count($transferred_pins) . ' PINs transferred successfully to ' . $to_username,
        'transferred_pins' => $transferred_pins
    ]);

} catch (Exception $e) {
    if (isset($connection) && $connection->ping()) {
        $connection->rollback();
    }
    
    error_log("PIN Transfer Error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while transferring PINs: ' . $e->getMessage()
    ]);
} 