<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to JSON
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Get search term
$search = trim($_GET['q'] ?? '');

if (empty($search)) {
    echo json_encode(['success' => true, 'users' => []]);
    exit;
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Search for users who have available (created/transferred) PINs
    $query = "SELECT DISTINCT u.id, u.username, u.full_name, u.email, 
                     COUNT(p.id) as available_pins 
              FROM users u 
              INNER JOIN pins p ON u.id = p.current_owner 
              WHERE (u.username LIKE ? OR u.full_name LIKE ? OR u.email LIKE ?) 
              AND p.status = 'created' 
              AND u.status = 'active' 
              AND u.id != ? 
              GROUP BY u.id 
              HAVING available_pins > 0 
              LIMIT 20";
    
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare query: " . $connection->error);
    }
    
    $searchParam = "%$search%";
    $currentUserId = $user->getId();
    $stmt->bind_param("sssi", $searchParam, $searchParam, $searchParam, $currentUserId);
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to execute query: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $users = [];
    
    while ($row = $result->fetch_assoc()) {
        $users[] = [
            'id' => $row['username'],
            'text' => sprintf(
                "%s - %s (%s) [%d available PINs]",
                $row['username'],
                $row['full_name'],
                $row['email'],
                $row['available_pins']
            )
        ];
    }
    
    echo json_encode([
        'success' => true,
        'users' => $users
    ]);

} catch (Exception $e) {
    error_log("User Search Error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while searching for users',
        'users' => []
    ]);
} 