<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../../index.php');
    exit;
}

// Get filter parameters
$search = trim($_GET['search'] ?? '');
$status = $_GET['status'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 10;
$offset = ($page - 1) * $limit;

// Calculate available PINs count (PINs owned by current user with 'created' status)
$db = Database::getInstance();
$availablePinsQuery = "SELECT COUNT(*) as count FROM pins WHERE current_owner = ? AND status = 'created'";
$stmt = $db->getConnection()->prepare($availablePinsQuery);
$userId = $user->getId();
$stmt->bind_param("i", $userId);
$stmt->execute();
$availablePinsCount = $stmt->get_result()->fetch_assoc()['count'];

// Build the base query
$baseQuery = "FROM pins p 
              LEFT JOIN users u1 ON p.created_by = u1.id 
              LEFT JOIN users u2 ON p.current_owner = u2.id 
              WHERE 1=1";

// Add search condition if provided
if ($search) {
    $baseQuery .= " AND (p.pin_number LIKE ? OR u1.username LIKE ? OR u2.username LIKE ?)";
    $params = ["%$search%", "%$search%", "%$search%"];
    $types = "sss";
} else {
    $params = [];
    $types = "";
}

// Add status condition if provided
if ($status) {
    $baseQuery .= " AND p.status = ?";
    $params[] = $status;
    $types .= "s";
}

// Get total records for pagination
$countQuery = "SELECT COUNT(*) as total " . $baseQuery;
$stmt = $db->getConnection()->prepare($countQuery);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$totalRecords = $stmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalRecords / $limit);

// Get data with pagination
$dataQuery = "SELECT p.*, 
                     u1.username as created_by_username, 
                     u2.username as current_owner_username,
                     u2.full_name as current_owner_name 
              " . $baseQuery . " 
              ORDER BY p.created_at DESC 
              LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= "ii";

$stmt = $db->getConnection()->prepare($dataQuery);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$pins = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - PIN Management</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(180deg, #3a86ff 0%, #8338ec 100%);
            padding-top: 20px;
            border-top-right-radius: 20px;
            border-bottom-right-radius: 20px;
            box-shadow: 2px 0 12px rgba(51,51,51,0.08);
            transition: background 0.3s;
        }
        .sidebar .sidebar-brand {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 2rem;
        }
        .sidebar .sidebar-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: #fff;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.5rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .sidebar .sidebar-logo i {
            font-size: 2rem;
            color: #3a86ff;
        }
        .sidebar .sidebar-title {
            color: #fff;
            font-weight: bold;
            font-size: 1.2rem;
            letter-spacing: 1px;
        }
        .sidebar-section {
            color: #e0e1dd;
            font-size: 0.85rem;
            text-transform: uppercase;
            margin: 1.5rem 0 0.5rem 1rem;
            letter-spacing: 0.08em;
        }
        .sidebar a {
            color: #fff;
            padding: 12px 22px;
            display: flex;
            align-items: center;
            border-radius: 8px 0 0 8px;
            margin-bottom: 6px;
            font-size: 1.05rem;
            transition: background 0.2s, color 0.2s, box-shadow 0.2s;
        }
        .sidebar a.active, .sidebar a:hover {
            background: rgba(255, 255, 255, 0.15);
            color: #ffbe0b;
            box-shadow: 2px 2px 8px rgba(0,0,0,0.07);
            font-weight: bold;
        }
        .sidebar a i {
            margin-right: 12px;
            font-size: 1.2em;
        }
        @media (max-width: 767.98px) {
            .sidebar {
                position: fixed;
                z-index: 1050;
                left: 0;
                top: 0;
                width: 220px;
                height: 100vh;
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                border-radius: 0 20px 20px 0;
            }
            .sidebar.collapse.show {
                transform: translateX(0);
            }
            .main-content {
                padding: 10px;
            }
        }
        .main-content {
            padding: 20px;
        }
        .status-badge {
            font-size: 0.875rem;
            padding: 5px 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar Overlay (for mobile) -->
            <div id="sidebarOverlay" style="display:none;position:fixed;top:0;left:0;width:100vw;height:100vh;background:rgba(0,0,0,0.3);z-index:1049;"></div>
            <!-- Sidebar Toggle Button (visible on mobile) -->
            <button class="btn btn-dark d-md-none m-2" type="button" data-toggle="collapse" data-target="#adminSidebar" aria-controls="adminSidebar" aria-expanded="false" aria-label="Toggle navigation">
                <i class="fas fa-bars"></i> Menu
            </button>
            <!-- Sidebar -->
            <nav id="adminSidebar" class="col-md-2 sidebar collapse d-md-block">
                <div class="sidebar-brand">
                    <div class="sidebar-logo">
                        <i class="fas fa-user-shield"></i>
                    </div>
                    <div class="sidebar-title"><?php echo BRAND_NAME; ?></div>
                    <div style="color:#e0e1dd;font-size:0.9em;">Admin Panel</div>
                </div>
                <div class="sidebar-section">Main</div>
                <a href="../dashboard.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                <div class="sidebar-section">Customers</div>
                <a href="../customers.php"><i class="fas fa-users"></i> Customers</a>
                <a href="../cus-products.php"><i class="fas fa-users"></i> Customers Products</a>
                <div class="sidebar-section">Management</div>
                <a href="index.php" class="active"><i class="fas fa-key"></i> PIN Management</a>
                <a href="../genealogy.php"><i class="fas fa-sitemap"></i> Genealogy</a>
                <a href="../reports/"><i class="fas fa-chart-bar"></i> Reports</a>
                <div class="sidebar-section">Support</div>
                <a href="../support/"><i class="fas fa-headset"></i> Support</a>
                <a href="../../logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h2>PIN Management</h2>
                    </div>
                    <div class="col-md-6 text-right">
                        <button class="btn btn-success mr-2" id="activatePinsBtn">
                            <i class="fas fa-check-circle mr-2"></i> Activate PINs
                        </button>
                        <button class="btn btn-success mr-2" id="transferPinsBtn">
                            <i class="fas fa-exchange-alt mr-2"></i> Transfer PINs
                        </button>
                        <button class="btn btn-primary" onclick="generatePins()">
                            <i class="fas fa-plus mr-2"></i> Generate PINs
                        </button>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" action="" class="row">
                            <div class="col-md-4">
                                <div class="form-group mb-0">
                                    <label>Search</label>
                                    <input type="text" name="search" class="form-control" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search PIN, username, or name...">
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group mb-0">
                                    <label>Status</label>
                                    <select name="status" class="form-control">
                                        <option value="">All Status</option>
                                        <option value="created" <?php echo $status === 'created' ? 'selected' : ''; ?>>Created</option>
                                        <option value="transferred" <?php echo $status === 'transferred' ? 'selected' : ''; ?>>Transferred</option>
                                        <option value="activated" <?php echo $status === 'activated' ? 'selected' : ''; ?>>Activated</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group mb-0">
                                    <label>&nbsp;</label>
                                    <button type="submit" class="btn btn-primary btn-block">
                                        <i class="fas fa-search mr-2"></i> Search
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group mb-0">
                                    <label>&nbsp;</label>
                                    <a href="index.php" class="btn btn-secondary btn-block">
                                        <i class="fas fa-redo mr-2"></i> Reset
                                    </a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- PINs Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped">
                                <thead>
                                    <tr>
                                        <th width="30">
                                            <input type="checkbox" id="selectAll" title="Select All">
                                        </th>
                                        <th>PIN Number</th>
                                        <th>Batch ID</th>
                                        <th>Created By</th>
                                        <th>Current Owner</th>
                                        <th>Status</th>
                                        <th>Created At</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($pins)): ?>
                                        <tr>
                                            <td colspan="8" class="text-center">No PINs found</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($pins as $pin): ?>
                                            <tr>
                                                <td>
                                                    <?php if ($pin['status'] === 'created' && $pin['current_owner'] === $user->getId()): ?>
                                                        <input type="checkbox" class="pin-checkbox" data-pin="<?php echo htmlspecialchars($pin['pin_number']); ?>">
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo htmlspecialchars($pin['pin_number']); ?></td>
                                                <td><?php echo htmlspecialchars($pin['batch_id']); ?></td>
                                                <td><?php echo htmlspecialchars($pin['created_by_username']); ?></td>
                                                <td>
                                                    <?php if ($pin['current_owner_username']): ?>
                                                        <a href="../customer_view.php?id=<?php echo $pin['current_owner']; ?>">
                                                            <?php echo htmlspecialchars($pin['current_owner_username']); ?>
                                                            <br>
                                                            <small class="text-muted"><?php echo htmlspecialchars($pin['current_owner_name']); ?></small>
                                                        </a>
                                                    <?php else: ?>
                                                        -
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="badge status-badge <?php 
                                                        echo $pin['status'] === 'created' ? 'badge-success' : 
                                                            ($pin['status'] === 'transferred' ? 'badge-primary' : 'badge-info'); 
                                                    ?>">
                                                        <?php echo ucfirst($pin['status']); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo date('M d, Y H:i', strtotime($pin['created_at'])); ?></td>
                                                <td>
                                                    <?php if ($pin['status'] === 'created'): ?>
                                                        <button class="btn btn-sm btn-primary mr-1" onclick="transferPin('<?php echo $pin['pin_number']; ?>')">
                                                            <i class="fas fa-exchange-alt"></i> Transfer
                                                        </button>
                                                        <button class="btn btn-sm btn-danger" onclick="expirePin('<?php echo $pin['pin_number']; ?>')">
                                                            <i class="fas fa-times"></i> Expire
                                                        </button>
                                                    <?php elseif ($pin['status'] === 'transferred'): ?>
                                                        <button class="btn btn-sm btn-success" onclick="activatePin('<?php echo $pin['pin_number']; ?>', '<?php echo $pin['current_owner_username']; ?>')">
                                                            <i class="fas fa-check"></i> Activate
                                                        </button>
                                                    <?php else: ?>
                                                        -
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($totalPages > 1): ?>
                            <nav aria-label="Page navigation" class="mt-4">
                                <ul class="pagination justify-content-center">
                                    <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($status); ?>">Previous</a>
                                    </li>
                                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                                        <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                            <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($status); ?>"><?php echo $i; ?></a>
                                        </li>
                                    <?php endfor; ?>
                                    <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($status); ?>">Next</a>
                                    </li>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Generate PINs Modal -->
    <div class="modal fade" id="generatePinsModal" tabindex="-1" role="dialog" aria-labelledby="generatePinsModalLabel" aria-hidden="true">
        <div class="modal-dialog" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="generatePinsModalLabel">Generate PINs</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <form onsubmit="return handlePinGeneration(event)">
                    <div class="modal-body">
                        <div class="form-group">
                            <label for="pinCount">Number of PINs to Generate (1-500)</label>
                            <input type="number" class="form-control" id="pinCount" name="count" min="1" max="500" value="10" required>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Generate</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Transfer PINs Modal -->
    <div class="modal fade" id="transferPinsModal" tabindex="-1" aria-labelledby="transferPinsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="transferPinsModalLabel">Transfer PINs</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <form id="transferPinsForm">
                        <div class="form-group">
                            <label class="font-weight-bold">Transfer Method</label>
                            <div class="custom-control custom-radio">
                                <input type="radio" id="methodSelection" name="transferMethod" value="selection" class="custom-control-input" checked>
                                <label class="custom-control-label" for="methodSelection">Selected PINs (Checkbox Selection)</label>
                            </div>
                            <div class="custom-control custom-radio">
                                <input type="radio" id="methodCount" name="transferMethod" value="count" class="custom-control-input">
                                <label class="custom-control-label" for="methodCount">By Count (Random Selection)</label>
                            </div>
                        </div>
                        
                        <div id="selectionMethod">
                            <div class="alert alert-info">
                                <span id="selectedPinsCount">0</span> PINs selected
                            </div>
                        </div>
                        
                        <div id="countMethod" style="display: none;">
                            <div class="form-group">
                                <label for="transferPinCount">Number of PINs to Transfer</label>
                                <input type="number" class="form-control" id="transferPinCount" min="1" max="<?php echo $availablePinsCount ?? 0; ?>">
                                <small class="form-text text-muted">Available PINs: <span id="availablePinsCount"><?php echo $availablePinsCount ?? 0; ?></span></small>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="transferUsername">Transfer to Username</label>
                            <select class="form-control" id="transferUsername" style="width: 100%"></select>
                            <small class="form-text text-muted">Search by username, name, email or phone</small>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" id="confirmTransferBtn">Transfer PINs</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Activate PINs Modal -->
    <div class="modal fade" id="activatePinsModal" tabindex="-1" aria-labelledby="activatePinsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="activatePinsModalLabel">Activate User with PIN</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <form id="activatePinForm">
                        <!-- Available PINs Display -->
                        <div class="form-group">
                            <label>Available PINs</label>
                            <select class="form-control" id="availablePin">
                                <option value="">Loading PINs...</option>
                            </select>
                            <small class="form-text text-muted">Select a PIN to use for activation</small>
                        </div>

                        <!-- User to Activate Selection -->
                        <div class="form-group">
                            <label for="userToActivate">User to Activate</label>
                            <select class="form-control" id="userToActivate" style="width: 100%"></select>
                            <small class="form-text text-muted">Search for an inactive user to activate</small>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" id="confirmActivationBtn" disabled>Activate User</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Success Toast -->
    <div class="position-fixed bottom-0 right-0 p-3" style="z-index: 5; right: 0; bottom: 0;">
        <div id="successToast" class="toast hide" role="alert" aria-live="assertive" aria-atomic="true" data-delay="5000">
            <div class="toast-header bg-success text-white">
                <strong class="mr-auto">Success</strong>
                <button type="button" class="ml-2 mb-1 close" data-dismiss="toast" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="toast-body" id="successToastMessage"></div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/popper.js@1.16.1/dist/umd/popper.min.js"></script>
    <script src="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        $(document).ready(function() {
            // Select All checkboxes
            $('#selectAll').change(function() {
                $('.pin-checkbox').prop('checked', $(this).prop('checked'));
                updateSelectedPinsCount();
            });
            
            // Update Select All when individual checkboxes change
            $(document).on('change', '.pin-checkbox', function() {
                if ($('.pin-checkbox:checked').length === $('.pin-checkbox').length) {
                    $('#selectAll').prop('checked', true);
                } else {
                    $('#selectAll').prop('checked', false);
                }
                updateSelectedPinsCount();
            });
            
            // Transfer PINs button click
            $('#transferPinsBtn').click(function() {
                updateSelectedPinsCount();
                $('#transferPinsModal').modal('show');
            });
            
            // Toggle transfer methods
            $('input[name="transferMethod"]').change(function() {
                if ($(this).val() === 'selection') {
                    $('#selectionMethod').show();
                    $('#countMethod').hide();
                } else {
                    $('#selectionMethod').hide();
                    $('#countMethod').show();
                }
            });
            
            function updateSelectedPinsCount() {
                var count = $('.pin-checkbox:checked').length;
                $('#selectedPinsCount').text(count);
            }
            
            // Initialize Select2 for username search
            $('#transferUsername').select2({
                ajax: {
                    url: 'search_users.php',
                    dataType: 'json',
                    delay: 250,
                    data: function(params) {
                        return {
                            q: params.term
                        };
                    },
                    processResults: function(data) {
                        return {
                            results: data.users
                        };
                    },
                    cache: true
                },
                placeholder: 'Search for a user',
                minimumInputLength: 2,
                dropdownParent: $('#transferPinsModal')
            });
            
            // Confirm Transfer button click
            $('#confirmTransferBtn').click(function() {
                var transferMethod = $('input[name="transferMethod"]:checked').val();
                var username = $('#transferUsername').val();
                
                if (!username) {
                    alert('Please select a user to transfer PINs to');
                    return;
                }
                
                var pins = [];
                var pinCount = 0;
                
                if (transferMethod === 'selection') {
                    $('.pin-checkbox:checked').each(function() {
                        pins.push($(this).data('pin'));
                    });
                    
                    if (pins.length === 0) {
                        alert('Please select at least one PIN to transfer');
                        return;
                    }
                } else {
                    pinCount = parseInt($('#transferPinCount').val());
                    
                    if (isNaN(pinCount) || pinCount <= 0) {
                        alert('Please enter a valid number of PINs to transfer');
                        return;
                    }
                }
                
                // Confirm transfer
                var confirmMessage = transferMethod === 'selection' 
                    ? 'Are you sure you want to transfer ' + pins.length + ' selected PINs?' 
                    : 'Are you sure you want to transfer ' + pinCount + ' random PINs?';
                
                if (confirm(confirmMessage)) {
                    transferPins(pins, username, pinCount, transferMethod);
                }
            });
            
            function transferPins(pins, username, pinCount, transferMethod) {
                var data = {
                    username: username
                };
                
                if (transferMethod === 'selection') {
                    data.pins = JSON.stringify(pins);
                } else {
                    data.count = pinCount;
                }
                
                $.ajax({
                    url: 'transfer.php',
                    type: 'POST',
                    data: data,
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            $('#transferPinsModal').modal('hide');
                            $('#successToastMessage').text(response.message);
                            $('#successToast').toast('show');
                            
                            // Reload page after a short delay
                            setTimeout(function() {
                                window.location.reload();
                            }, 2000);
                        } else {
                            alert(response.message || 'An error occurred while transferring PINs');
                        }
                    },
                    error: function() {
                        alert('An error occurred while transferring PINs');
                    }
                });
            }

            // Initialize Select2 for inactive user search
            $('#userToActivate').select2({
                ajax: {
                    url: 'search_inactive_users.php',
                    dataType: 'json',
                    delay: 250,
                    data: function(params) {
                        return {
                            q: params.term
                        };
                    },
                    processResults: function(data) {
                        return {
                            results: data.users
                        };
                    },
                    cache: true
                },
                placeholder: 'Search for inactive users',
                minimumInputLength: 2,
                dropdownParent: $('#activatePinsModal')
            });

            // Show Activate PINs modal and load available PINs
            $('#activatePinsBtn').click(function() {
                // Reset form
                $('#userToActivate').val(null).trigger('change');
                $('#availablePin').empty().append('<option value="">Loading PINs...</option>');
                $('#confirmActivationBtn').prop('disabled', true);
                
                // Fetch available PINs for the current user
                $.ajax({
                    url: 'get_available_pins.php',
                    type: 'GET',
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            var $pinSelect = $('#availablePin');
                            $pinSelect.empty().append('<option value="">Select a PIN</option>');
                            
                            response.pins.forEach(function(pin) {
                                $pinSelect.append(
                                    $('<option></option>')
                                        .val(pin.id)
                                        .text(pin.pin_code + ' (Created: ' + pin.created_at + ')')
                                );
                            });
                        } else {
                            alert(response.message || 'Failed to fetch available PINs');
                            $('#activatePinsModal').modal('hide');
                        }
                    },
                    error: function() {
                        alert('An error occurred while fetching available PINs');
                        $('#activatePinsModal').modal('hide');
                    }
                });
                
                $('#activatePinsModal').modal('show');
            });

            // Handle form validation
            function validateActivationForm() {
                var selectedPin = $('#availablePin').val();
                var userToActivate = $('#userToActivate').val();
                
                $('#confirmActivationBtn').prop('disabled', 
                    !selectedPin || !userToActivate
                );
            }

            // Add change event listeners for validation
            $('#availablePin, #userToActivate').on('change', validateActivationForm);

            // Handle activation confirmation
            $('#confirmActivationBtn').click(function() {
                if (confirm('Are you sure you want to activate this user? This action cannot be undone.')) {
                    var data = {
                        pin_id: $('#availablePin').val(),
                        target_username: $('#userToActivate').val()
                    };

                    // Show loading state
                    var $btn = $(this);
                    $btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Processing...');

                    $.ajax({
                        url: 'activate.php',
                        type: 'POST',
                        data: data,
                        dataType: 'json',
                        success: function(response) {
                            if (response.success) {
                                $('#activatePinsModal').modal('hide');
                                $('#successToastMessage').text(response.message);
                                $('#successToast').toast('show');
                                
                                // Reload page after a short delay
                                setTimeout(function() {
                                    window.location.reload();
                                }, 2000);
                            } else {
                                alert(response.message || 'Failed to activate user');
                                // Reset button state
                                $btn.prop('disabled', false).html('Activate User');
                            }
                        },
                        error: function() {
                            alert('An error occurred while activating the user');
                            // Reset button state
                            $btn.prop('disabled', false).html('Activate User');
                        }
                    });
                }
            });
        });

        function generatePins() {
            $('#generatePinsModal').modal('show');
        }

        function handlePinGeneration(event) {
            event.preventDefault();
            const form = event.target;
            const count = form.count.value;

            $.ajax({
                url: 'generate_pins.php',
                method: 'POST',
                data: { count: count },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        alert('PINs generated successfully');
                        window.location.reload();
                    } else {
                        alert(response.message || 'An error occurred');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error:', xhr.responseText);
                    alert('An error occurred while generating PINs. Please check the console for details.');
                }
            });

            $('#generatePinsModal').modal('hide');
            return false;
        }

        function transferPin(pin) {
            $('#transferPinNumber').val(pin);
            $('#transferPinModal').modal('show');
        }

        function handlePinTransfer(event) {
            event.preventDefault();
            const form = event.target;
            const pin = form.pin.value;
            const username = form.username.value;

            $.ajax({
                url: 'transfer.php',
                method: 'POST',
                data: { 
                    pins: JSON.stringify([pin]),
                    username: username 
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        alert('PIN transferred successfully');
                        window.location.reload();
                    } else {
                        alert(response.message || 'An error occurred');
                    }
                },
                error: function() {
                    alert('An error occurred while transferring the PIN');
                }
            });

            $('#transferPinModal').modal('hide');
            return false;
        }

        function activatePin(pin, username) {
            $('#activatePinNumber').val(pin);
            $('#activateUsername').val(username);
            $('#activateUsernameDisplay').text(username);
            $('#activatePinModal').modal('show');
        }

        function handlePinActivation(event) {
            event.preventDefault();
            const form = event.target;
            const pin = form.pin.value;
            const username = form.username.value;

            $.ajax({
                url: 'activate.php',
                method: 'POST',
                data: { pin: pin, username: username },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        alert('PIN activated successfully');
                        window.location.reload();
                    } else {
                        alert(response.message || 'An error occurred');
                    }
                },
                error: function() {
                    alert('An error occurred while activating the PIN');
                }
            });

            $('#activatePinModal').modal('hide');
            return false;
        }

        function expirePin(pin) {
            if (confirm('Are you sure you want to expire this PIN?')) {
                $.ajax({
                    url: 'expire_pin.php',
                    method: 'POST',
                    data: { pin: pin },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            alert('PIN expired successfully');
                            window.location.reload();
                        } else {
                            alert(response.message || 'An error occurred');
                        }
                    },
                    error: function() {
                        alert('An error occurred while expiring the PIN');
                    }
                });
            }
        }

        $(function() {
            var sidebar = $('#adminSidebar');
            var overlay = $('#sidebarOverlay');
            sidebar.on('show.bs.collapse', function() {
                if (window.innerWidth < 768) overlay.show();
            });
            sidebar.on('hide.bs.collapse', function() {
                overlay.hide();
            });
            overlay.on('click', function() {
                sidebar.collapse('hide');
            });
            $(window).on('resize', function() {
                if (window.innerWidth >= 768) overlay.hide();
            });
        });
    </script>
</body>
</html> 