<?php
require_once __DIR__ . '/../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get the number of PINs to generate
$count = intval($_POST['count'] ?? 0);

// Validate input
if ($count < 1 || $count > 500) {
    echo json_encode(['success' => false, 'message' => 'Invalid number of PINs requested (1-500)']);
    exit;
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Verify database connection
    if ($connection->connect_error) {
        throw new Exception("Database connection failed: " . $connection->connect_error);
    }
    
    // Verify we're using the correct database
    $result = $connection->query("SELECT DATABASE()");
    if (!$result) {
        throw new Exception("Failed to check database: " . $connection->error);
    }
    
    $dbName = $result->fetch_row()[0];
    if ($dbName !== 'zealmart') {
        throw new Exception("Wrong database in use: " . $dbName);
    }
    
    // Start transaction
    if (!$connection->begin_transaction()) {
        throw new Exception("Failed to start transaction: " . $connection->error);
    }

    // Verify PIN_PREFIX constant
    if (!defined('PIN_PREFIX')) {
        throw new Exception("PIN_PREFIX constant is not defined");
    }

    $pins = [];
    $insertQuery = "INSERT INTO pins (pin_number, batch_id, created_by, current_owner, status) VALUES (?, ?, ?, ?, 'created')";
    $stmt = $connection->prepare($insertQuery);
    
    if (!$stmt) {
        throw new Exception("Failed to prepare insert statement: " . $connection->error);
    }

    // Generate batch ID
    $batchId = 'BATCH-' . date('YmdHis') . '-' . strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 6));
    $adminId = $user->getId();
    
    if (!$adminId) {
        throw new Exception("Failed to get admin user ID");
    }

    // Generate and insert PINs
    for ($i = 0; $i < $count; $i++) {
        $attempts = 0;
        $maxAttempts = 10;
        $pinGenerated = false;

        while (!$pinGenerated && $attempts < $maxAttempts) {
            try {
                // Generate a random PIN in the format Z98765432A
                $pin = PIN_PREFIX . mt_rand(10000000, 99999999) . chr(mt_rand(65, 90));

                // Check if PIN already exists
                $checkQuery = "SELECT COUNT(*) as count FROM pins WHERE pin_number = ?";
                $checkStmt = $connection->prepare($checkQuery);
                if (!$checkStmt) {
                    throw new Exception("Failed to prepare check statement: " . $connection->error);
                }
                
                $checkStmt->bind_param("s", $pin);
                if (!$checkStmt->execute()) {
                    throw new Exception("Failed to check PIN existence: " . $checkStmt->error);
                }
                
                $result = $checkStmt->get_result()->fetch_assoc();
                
                if ($result['count'] === 0) {
                    // PIN is unique, insert it
                    if (!$stmt->bind_param("ssii", $pin, $batchId, $adminId, $adminId)) {
                        throw new Exception("Failed to bind parameters: " . $stmt->error);
                    }
                    
                    if (!$stmt->execute()) {
                        throw new Exception("Failed to insert PIN: " . $stmt->error);
                    }

                    $pins[] = $pin;
                    $pinGenerated = true;
                }
            } catch (Exception $e) {
                $attempts++;
                if ($attempts >= $maxAttempts) {
                    throw new Exception("Failed to generate unique PIN after {$maxAttempts} attempts: " . $e->getMessage());
                }
            }
        }
    }

    // Commit transaction
    if (!$connection->commit()) {
        throw new Exception("Failed to commit transaction: " . $connection->error);
    }

    echo json_encode([
        'success' => true,
        'message' => 'PINs generated successfully',
        'pins' => $pins,
        'batch_id' => $batchId
    ]);

} catch (Exception $e) {
    // Rollback transaction if it's active
    if (isset($connection) && $connection->ping()) {
        $connection->rollback();
    }
    
    error_log("PIN Generation Error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while generating PINs: ' . $e->getMessage()
    ]);
} 