<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to JSON
header('Content-Type: application/json');

// Check if user is logged in
$user = new User();
if (!$user->isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get POST data
$pinId = $_POST['pin_id'] ?? '';
$targetUsername = $_POST['target_username'] ?? '';

if (empty($pinId) || empty($targetUsername)) {
    echo json_encode(['success' => false, 'message' => 'PIN ID and target username are required']);
    exit;
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Start transaction
    $connection->begin_transaction();
    
    // Get target user details
    $query = "SELECT id, status FROM users WHERE username = ?";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare user query: " . $connection->error);
    }
    
    $stmt->bind_param("s", $targetUsername);
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to execute user query: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $targetUser = $result->fetch_assoc();
    
    if (!$targetUser) {
        throw new Exception("Target user not found");
    }
    
    if ($targetUser['status'] !== 'inactive') {
        throw new Exception("Target user is already active");
    }
    
    // Get PIN details
    $query = "SELECT id, current_owner, status FROM pins WHERE id = ?";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare PIN query: " . $connection->error);
    }
    
    $stmt->bind_param("i", $pinId);
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to execute PIN query: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $pin = $result->fetch_assoc();
    
    if (!$pin) {
        throw new Exception("PIN not found");
    }
    
    if ($pin['status'] !== 'created') {
        throw new Exception("PIN is already used");
    }
    
    // Update PIN status
    $query = "UPDATE pins SET status = 'activated', current_owner = ?, activated_at = NOW() WHERE id = ?";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare PIN update query: " . $connection->error);
    }
    
    $stmt->bind_param("ii", $targetUser['id'], $pinId);
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to update PIN: " . $stmt->error);
    }
    
    // Activate user
    $query = "UPDATE users SET status = 'active', updated_at = NOW() WHERE id = ?";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare user update query: " . $connection->error);
    }
    
    $stmt->bind_param("i", $targetUser['id']);
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to activate user: " . $stmt->error);
    }
    
    // Fetch parent_id from registration (assume stored in a custom field or genealogy pending table, or fetch from genealogy if user was pre-placed)
    // For this implementation, try to find the parent by looking for a genealogy entry where this user is a child (pre-placed), else NULL
    $parent_id = null;
    $query = "SELECT parent_id FROM genealogy WHERE user_id = ? AND parent_id IS NOT NULL LIMIT 1";
    $stmt = $connection->prepare($query);
    if ($stmt) {
        $stmt->bind_param("i", $targetUser['id']);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            if ($result && $row = $result->fetch_assoc()) {
                $parent_id = $row['parent_id'];
            }
        }
    }

    // Remove any existing genealogy entry for this user (cleanup)
    $query = "DELETE FROM genealogy WHERE user_id = ?";
    $stmt = $connection->prepare($query);
    if ($stmt) {
    $stmt->bind_param("i", $targetUser['id']);
        $stmt->execute();
    }

    // Insert genealogy with parent_id
    if ($parent_id) {
        $query = "INSERT INTO genealogy (user_id, parent_id, placement_level, position) VALUES (?, ?, 1, 1)";
        $stmt = $connection->prepare($query);
        if (!$stmt) {
            throw new Exception("Failed to prepare genealogy query: " . $connection->error);
        }
        $stmt->bind_param("ii", $targetUser['id'], $parent_id);
    } else {
        $query = "INSERT INTO genealogy (user_id, parent_id, placement_level, position) VALUES (?, NULL, 1, 1)";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
            throw new Exception("Failed to prepare genealogy query: " . $connection->error);
        }
        $stmt->bind_param("i", $targetUser['id']);
    }
    if (!$stmt->execute()) {
        throw new Exception("Failed to create genealogy entry: " . $stmt->error . ' | parent_id: ' . var_export($parent_id, true));
    }

    // MLM Earnings Distribution Logic (REVISED)
    $mlm_plan = [
        1 => ['members' => 3, 'payout' => 250],
        2 => ['members' => 9, 'payout' => 500],
        3 => ['members' => 27, 'payout' => 1000],
        4 => ['members' => 81, 'payout' => 2000],
        5 => ['members' => 243, 'payout' => 4000],
        6 => ['members' => 729, 'payout' => 8000],
        7 => ['members' => 2187, 'payout' => 16000],
        8 => ['members' => 6561, 'payout' => 32000],
        9 => ['members' => 19683, 'payout' => 64000],
        10 => ['members' => 59049, 'payout' => 128000],
        11 => ['members' => 177147, 'payout' => 256000],
        12 => ['members' => 531441, 'payout' => 512000],
        13 => ['members' => 1594323, 'payout' => 1024000],
        14 => ['members' => 4782969, 'payout' => 2048000],
        15 => ['members' => 14348907, 'payout' => 4096000],
        16 => ['members' => 43046721, 'payout' => 8192000],
        17 => ['members' => 129140163, 'payout' => 16384000],
        18 => ['members' => 387420489, 'payout' => 32768000],
        19 => ['members' => 1162261467, 'payout' => 65536000],
        20 => ['members' => 3486784401, 'payout' => 131072000],
    ];

    function get_downline_at_level($connection, $user_id, $level) {
        $current_level = 0;
        $current_ids = [$user_id];
        while ($current_level < $level) {
            if (empty($current_ids)) return [];
            $placeholders = implode(',', array_fill(0, count($current_ids), '?'));
            $types = str_repeat('i', count($current_ids));
            $stmt = $connection->prepare("SELECT user_id FROM genealogy WHERE parent_id IN ($placeholders)");
            $stmt->bind_param($types, ...$current_ids);
            $stmt->execute();
            $result = $stmt->get_result();
            $next_ids = [];
            while ($row = $result->fetch_assoc()) {
                $next_ids[] = $row['user_id'];
            }
            $current_ids = $next_ids;
            $current_level++;
        }
        return $current_ids;
    }

    // For each upline, check if they qualify for a payout at any level
    $current_user_id = $targetUser['id'];
    for ($level = 1; $level <= 20; $level++) {
        // Get parent
        $query = "SELECT parent_id FROM genealogy WHERE user_id = ?";
        $stmt = $connection->prepare($query);
        if (!$stmt) break;
        $stmt->bind_param("i", $current_user_id);
        if (!$stmt->execute()) break;
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        if (!$row || !$row['parent_id']) break;
        $upline_id = $row['parent_id'];
        // Check if upline qualifies for payout at this level
        $downline = get_downline_at_level($connection, $upline_id, $level);
        if (count($downline) >= $mlm_plan[$level]['members']) {
            // Check if earning already exists
            $check = $connection->prepare("SELECT id FROM earnings WHERE user_id = ? AND amount = ? AND earning_type = 'cash'");
            $check->bind_param("id", $upline_id, $mlm_plan[$level]['payout']);
            $check->execute();
            $check->store_result();
            if ($check->num_rows == 0) {
                // Insert earning
                $ins = $connection->prepare("INSERT INTO earnings (user_id, amount, earning_type, status) VALUES (?, ?, 'cash', 'available')");
                $ins->bind_param("id", $upline_id, $mlm_plan[$level]['payout']);
                $ins->execute();
            }
        }
        $current_user_id = $upline_id;
    }
    // END MLM Earnings Distribution Logic
    
    // Commit transaction
    $connection->commit();
    
    echo json_encode([
        'success' => true,
        'message' => "Successfully activated user {$targetUsername}"
    ]);

} catch (Exception $e) {
    // Rollback transaction on error
    if (isset($connection) && $connection->ping()) {
        $connection->rollback();
    }
    
    error_log("PIN Activation Error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
} 