<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get POST data
$user_id = intval($_POST['user_id'] ?? 0);
$action = trim($_POST['action'] ?? ''); // 'verify' or 'reject'
$remarks = trim($_POST['remarks'] ?? '');

if (!$user_id || !in_array($action, ['verify', 'reject'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid parameters']);
    exit;
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Start transaction
    if (!$connection->begin_transaction()) {
        throw new Exception("Failed to start transaction");
    }

    // Update KYC status
    $status = $action === 'verify' ? 'verified' : 'rejected';
    $query = "UPDATE user_profiles SET kyc_status = ?, kyc_remarks = ?, verified_at = CURRENT_TIMESTAMP WHERE user_id = ?";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare query: " . $connection->error);
    }
    
    $stmt->bind_param("ssi", $status, $remarks, $user_id);
    if (!$stmt->execute()) {
        throw new Exception("Failed to update KYC status: " . $stmt->error);
    }

    // If verifying, update user status to active
    if ($action === 'verify') {
        $query = "UPDATE users SET status = 'active' WHERE id = ?";
        $stmt = $connection->prepare($query);
        if (!$stmt) {
            throw new Exception("Failed to prepare user update query: " . $connection->error);
        }
        
        $stmt->bind_param("i", $user_id);
        if (!$stmt->execute()) {
            throw new Exception("Failed to update user status: " . $stmt->error);
        }
    }

    // Commit transaction
    if (!$connection->commit()) {
        throw new Exception("Failed to commit transaction");
    }

    echo json_encode([
        'success' => true,
        'message' => 'KYC ' . ($action === 'verify' ? 'verified' : 'rejected') . ' successfully'
    ]);

} catch (Exception $e) {
    if (isset($connection) && $connection->ping()) {
        $connection->rollback();
    }
    
    error_log("KYC Verification Error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while processing KYC: ' . $e->getMessage()
    ]);
} 