<?php
require_once '../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../index.php');
    exit;
}

// Get customer ID
$customer_id = intval($_GET['id'] ?? 0);
if (!$customer_id) {
    header('Location: customers.php');
    exit;
}

// Get customer data
$db = Database::getInstance();
$query = "SELECT u.*, 
          COALESCE(SUM(CASE WHEN e.status = 'available' THEN e.amount ELSE 0 END), 0) as available_earnings,
          COALESCE(SUM(e.amount), 0) as total_earnings,
          up.pan_number, up.id_proof_type, up.id_proof_number, up.kyc_status,
          bd.bank_name, bd.account_number, bd.account_type, bd.ifsc_code, bd.branch_name,
          bd.upi_id, bd.upi_number, bd.upi_app
          FROM users u 
          LEFT JOIN earnings e ON u.id = e.user_id 
          LEFT JOIN user_profiles up ON u.id = up.user_id
          LEFT JOIN banking_details bd ON u.id = bd.user_id
          WHERE u.id = ? AND u.role = 'user'
          GROUP BY u.id";

$stmt = $db->getConnection()->prepare($query);
$stmt->bind_param("i", $customer_id);
$stmt->execute();
$customer = $stmt->get_result()->fetch_assoc();

if (!$customer) {
    header('Location: customers.php');
    exit;
}

// Get referral data
$referralQuery = "SELECT COUNT(*) as direct_referrals FROM genealogy WHERE parent_id = ? AND placement_level = 1";
$stmt = $db->getConnection()->prepare($referralQuery);
$stmt->bind_param("i", $customer_id);
$stmt->execute();
$referralData = $stmt->get_result()->fetch_assoc();

// Get team size
$teamQuery = "SELECT COUNT(*) as team_size FROM genealogy WHERE parent_id = ?";
$stmt = $db->getConnection()->prepare($teamQuery);
$stmt->bind_param("i", $customer_id);
$stmt->execute();
$teamData = $stmt->get_result()->fetch_assoc();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Customer Details</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
        .info-card {
            background-color: #f8f9fa;
            border-radius: 5px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .status-badge {
            font-size: 1rem;
            padding: 5px 15px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="customers.php" class="active"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="reports/"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="support/"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h2>Customer Details</h2>
                    </div>
                    <div class="col-md-6 text-right">
                        <a href="customers.php" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> Back to List
                        </a>
                        <button class="btn btn-primary" onclick="editCustomer(<?php echo $customer_id; ?>)">
                            <i class="fas fa-edit"></i> Edit Customer
                        </button>
                    </div>
                </div>

                <!-- Customer Overview -->
                <div class="row">
                    <div class="col-md-8">
                        <div class="card mb-4">
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <h4><?php echo htmlspecialchars($customer['full_name']); ?></h4>
                                        <p class="text-muted mb-0">Username: <?php echo htmlspecialchars($customer['username']); ?></p>
                                        <p class="text-muted mb-0">Email: <?php echo htmlspecialchars($customer['email']); ?></p>
                                        <p class="text-muted">Phone: <?php echo htmlspecialchars($customer['phone']); ?></p>
                                    </div>
                                    <div class="col-md-6 text-right">
                                        <span class="badge <?php echo $customer['status'] === 'active' ? 'badge-success' : 'badge-danger'; ?> status-badge">
                                            <?php echo ucfirst($customer['status'] ?? 'unknown'); ?>
                                        </span>
                                        <p class="mt-2">Member since: <?php echo date('M d, Y', strtotime($customer['created_at'])); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Statistics Cards -->
                        <div class="row">
                            <div class="col-md-3">
                                <div class="info-card">
                                    <h5>Total Earnings</h5>
                                    <h3>₹<?php echo number_format($customer['total_earnings'] ?? 0, 2); ?></h3>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="info-card">
                                    <h5>Available</h5>
                                    <h3>₹<?php echo number_format($customer['available_earnings'] ?? 0, 2); ?></h3>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="info-card">
                                    <h5>Direct Referrals</h5>
                                    <h3><?php echo $referralData['direct_referrals'] ?? 0; ?></h3>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="info-card">
                                    <h5>Team Size</h5>
                                    <h3><?php echo $teamData['team_size'] ?? 0; ?></h3>
                                </div>
                            </div>
                        </div>

                        <!-- KYC Details -->
                        <div class="card mt-4">
                            <div class="card-header">
                                <h5 class="mb-0">KYC Details</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>PAN Number:</strong></p>
                                        <p class="text-muted"><?php echo $customer['pan_number'] ?: 'Not provided'; ?></p>
                                    </div>
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>KYC Status:</strong></p>
                                        <span class="badge <?php 
                                            echo ($customer['kyc_status'] ?? '') === 'verified' ? 'badge-success' : 
                                                (($customer['kyc_status'] ?? '') === 'rejected' ? 'badge-danger' : 'badge-warning'); 
                                        ?>">
                                            <?php echo ucfirst($customer['kyc_status'] ?? 'pending'); ?>
                                        </span>
                                    </div>
                                </div>
                                <div class="row mt-3">
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>ID Proof Type:</strong></p>
                                        <p class="text-muted"><?php echo $customer['id_proof_type'] ?: 'Not provided'; ?></p>
                                    </div>
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>ID Proof Number:</strong></p>
                                        <p class="text-muted"><?php echo $customer['id_proof_number'] ?: 'Not provided'; ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Banking Details -->
                        <div class="card mt-4">
                            <div class="card-header">
                                <h5 class="mb-0">Banking Details</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>Bank Name:</strong></p>
                                        <p class="text-muted"><?php echo $customer['bank_name'] ?: 'Not provided'; ?></p>
                                        
                                        <p class="mb-1"><strong>Account Number:</strong></p>
                                        <p class="text-muted"><?php echo $customer['account_number'] ?: 'Not provided'; ?></p>
                                        
                                        <p class="mb-1"><strong>Account Type:</strong></p>
                                        <p class="text-muted"><?php echo $customer['account_type'] ?: 'Not provided'; ?></p>
                                    </div>
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>IFSC Code:</strong></p>
                                        <p class="text-muted"><?php echo $customer['ifsc_code'] ?: 'Not provided'; ?></p>
                                        
                                        <p class="mb-1"><strong>Branch Name:</strong></p>
                                        <p class="text-muted"><?php echo $customer['branch_name'] ?: 'Not provided'; ?></p>
                                    </div>
                                </div>
                                <hr>
                                <div class="row">
                                    <div class="col-md-4">
                                        <p class="mb-1"><strong>UPI ID:</strong></p>
                                        <p class="text-muted"><?php echo $customer['upi_id'] ?: 'Not provided'; ?></p>
                                    </div>
                                    <div class="col-md-4">
                                        <p class="mb-1"><strong>UPI Number:</strong></p>
                                        <p class="text-muted"><?php echo $customer['upi_number'] ?: 'Not provided'; ?></p>
                                    </div>
                                    <div class="col-md-4">
                                        <p class="mb-1"><strong>UPI App:</strong></p>
                                        <p class="text-muted"><?php echo $customer['upi_app'] ?: 'Not provided'; ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Right Sidebar -->
                    <div class="col-md-4">
                        <!-- Quick Actions -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="mb-0">Quick Actions</h5>
                            </div>
                            <div class="card-body">
                                <div class="list-group">
                                    <button class="list-group-item list-group-item-action" onclick="changePassword(<?php echo $customer_id; ?>)">
                                        <i class="fas fa-key mr-2"></i> Change Password
                                    </button>
                                    <button class="list-group-item list-group-item-action" onclick="toggleStatus(<?php echo $customer_id; ?>)">
                                        <i class="fas fa-toggle-on mr-2"></i> Toggle Status
                                    </button>
                                    <button class="list-group-item list-group-item-action" onclick="viewGenealogy(<?php echo $customer_id; ?>)">
                                        <i class="fas fa-sitemap mr-2"></i> View Genealogy
                                    </button>
                                    <button class="list-group-item list-group-item-action" onclick="viewEarnings(<?php echo $customer_id; ?>)">
                                        <i class="fas fa-money-bill-wave mr-2"></i> View Earnings
                                    </button>
                                    <button class="list-group-item list-group-item-action" onclick="managePins(<?php echo $customer_id; ?>)">
                                        <i class="fas fa-key mr-2"></i> Manage PINs
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Recent Activities -->
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Recent Activities</h5>
                            </div>
                            <div class="card-body">
                                <p class="text-muted">No recent activities to display.</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function editCustomer(id) {
            window.location.href = `customer_edit.php?id=${id}`;
        }

        function changePassword(id) {
            window.location.href = `change_password.php?id=${id}`;
        }

        function toggleStatus(id) {
            const currentStatus = document.querySelector('.status-badge').textContent.toLowerCase();
            const newStatus = currentStatus === 'active' ? 'inactive' : 'active';

            if (confirm(`Are you sure you want to ${currentStatus === 'active' ? 'deactivate' : 'activate'} this customer?`)) {
                $.ajax({
                    url: 'toggle_status.php',
                    method: 'POST',
                    data: { id: id, status: newStatus },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            // Update the status badge
                            const badge = document.querySelector('.status-badge');
                            badge.textContent = response.new_status.charAt(0).toUpperCase() + response.new_status.slice(1);
                            badge.className = `badge status-badge ${response.new_status === 'active' ? 'badge-success' : 'badge-danger'}`;
                            
                            // Show success message
                            alert('Status updated successfully');
                        } else {
                            alert(response.message || 'An error occurred');
                        }
                    },
                    error: function() {
                        alert('An error occurred while updating the status');
                    }
                });
            }
        }

        function viewGenealogy(id) {
            window.location.href = `genealogy.php?user_id=${id}`;
        }

        function viewEarnings(id) {
            window.location.href = `reports/earnings.php?user_id=${id}`;
        }

        function managePins(id) {
            window.location.href = `pins/index.php?user_id=${id}`;
        }
    </script>
</body>
</html> 