<?php
require_once '../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../index.php');
    exit;
}

// Initialize variables
$errors = [];
$success = false;

// Get customer ID
$customer_id = intval($_GET['id'] ?? 0);
if (!$customer_id) {
    header('Location: customers.php');
    exit;
}

// Get customer data
$db = Database::getInstance();
$query = "SELECT u.*, 
          up.pan_number, up.id_proof_type, up.id_proof_number, up.kyc_status,
          bd.bank_name, bd.account_number, bd.account_type, bd.ifsc_code, bd.branch_name,
          bd.upi_id, bd.upi_number, bd.upi_app
          FROM users u 
          LEFT JOIN user_profiles up ON u.id = up.user_id
          LEFT JOIN banking_details bd ON u.id = bd.user_id
          WHERE u.id = ? AND u.role = 'user'";

$stmt = $db->getConnection()->prepare($query);
$stmt->bind_param("i", $customer_id);
$stmt->execute();
$customer = $stmt->get_result()->fetch_assoc();

if (!$customer) {
    header('Location: customers.php');
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate and sanitize input
    $full_name = trim($_POST['full_name'] ?? '');
    $email = filter_var(trim($_POST['email'] ?? ''), FILTER_SANITIZE_EMAIL);
    $phone = trim($_POST['phone'] ?? '');
    $status = $_POST['status'] ?? '';
    
    // KYC details
    $pan_number = trim($_POST['pan_number'] ?? '');
    $id_proof_type = trim($_POST['id_proof_type'] ?? '');
    $id_proof_number = trim($_POST['id_proof_number'] ?? '');
    $kyc_status = $_POST['kyc_status'] ?? '';

    // Banking details
    $bank_name = trim($_POST['bank_name'] ?? '');
    $account_number = trim($_POST['account_number'] ?? '');
    $account_type = trim($_POST['account_type'] ?? '');
    $ifsc_code = trim($_POST['ifsc_code'] ?? '');
    $branch_name = trim($_POST['branch_name'] ?? '');
    $upi_id = trim($_POST['upi_id'] ?? '');
    $upi_number = trim($_POST['upi_number'] ?? '');
    $upi_app = trim($_POST['upi_app'] ?? '');

    // Validate required fields
    if (empty($full_name)) $errors[] = "Full name is required";
    if (empty($email)) $errors[] = "Email is required";
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = "Invalid email format";
    if (empty($phone)) $errors[] = "Phone number is required";
    if (empty($status)) $errors[] = "Status is required";

    // If no errors, proceed with update
    if (empty($errors)) {
        try {
            $db->getConnection()->begin_transaction();

            // Update users table
            $updateUser = "UPDATE users SET 
                          full_name = ?, 
                          email = ?, 
                          phone = ?, 
                          status = ?
                          WHERE id = ?";
            $stmt = $db->getConnection()->prepare($updateUser);
            $stmt->bind_param("ssssi", $full_name, $email, $phone, $status, $customer_id);
            $result = $stmt->execute();
            if (!$result) {
                throw new Exception("Failed to update user: " . $stmt->error);
            }

            // Update user_profiles table
            $checkProfile = "SELECT COUNT(*) as count FROM user_profiles WHERE user_id = ?";
            $stmt = $db->getConnection()->prepare($checkProfile);
            $stmt->bind_param("i", $customer_id);
            $stmt->execute();
            $profileExists = $stmt->get_result()->fetch_assoc()['count'] > 0;

            if ($profileExists) {
                $updateProfile = "UPDATE user_profiles SET 
                                pan_number = ?,
                                id_proof_type = ?,
                                id_proof_number = ?,
                                kyc_status = ?
                                WHERE user_id = ?";
                $stmt = $db->getConnection()->prepare($updateProfile);
                $stmt->bind_param("ssssi", $pan_number, $id_proof_type, $id_proof_number, $kyc_status, $customer_id);
            } else {
                $updateProfile = "INSERT INTO user_profiles 
                                (user_id, pan_number, id_proof_type, id_proof_number, kyc_status)
                                VALUES (?, ?, ?, ?, ?)";
                $stmt = $db->getConnection()->prepare($updateProfile);
                $stmt->bind_param("issss", $customer_id, $pan_number, $id_proof_type, $id_proof_number, $kyc_status);
            }
            $result = $stmt->execute();
            if (!$result) {
                throw new Exception("Failed to update profile: " . $stmt->error);
            }

            // Update banking_details table
            $checkBanking = "SELECT COUNT(*) as count FROM banking_details WHERE user_id = ?";
            $stmt = $db->getConnection()->prepare($checkBanking);
            $stmt->bind_param("i", $customer_id);
            $stmt->execute();
            $bankingExists = $stmt->get_result()->fetch_assoc()['count'] > 0;

            if ($bankingExists) {
                $updateBanking = "UPDATE banking_details SET 
                                bank_name = ?,
                                account_number = ?,
                                account_type = ?,
                                ifsc_code = ?,
                                branch_name = ?,
                                upi_id = ?,
                                upi_number = ?,
                                upi_app = ?
                                WHERE user_id = ?";
                $stmt = $db->getConnection()->prepare($updateBanking);
                $stmt->bind_param("ssssssssi", $bank_name, $account_number, $account_type, 
                                $ifsc_code, $branch_name, $upi_id, $upi_number, $upi_app, $customer_id);
            } else {
                $updateBanking = "INSERT INTO banking_details 
                                (user_id, bank_name, account_number, account_type, ifsc_code, 
                                branch_name, upi_id, upi_number, upi_app)
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt = $db->getConnection()->prepare($updateBanking);
                $stmt->bind_param("issssssss", $customer_id, $bank_name, $account_number, $account_type, 
                                $ifsc_code, $branch_name, $upi_id, $upi_number, $upi_app);
            }
            $result = $stmt->execute();
            if (!$result) {
                throw new Exception("Failed to update banking details: " . $stmt->error);
            }

            $db->getConnection()->commit();
            $success = true;
            
            // Refresh customer data
            $stmt = $db->getConnection()->prepare($query);
            $stmt->bind_param("i", $customer_id);
            $stmt->execute();
            $customer = $stmt->get_result()->fetch_assoc();

        } catch (Exception $e) {
            $db->getConnection()->rollback();
            $errors[] = "An error occurred while updating the customer: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Edit Customer</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="customers.php" class="active"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="reports/"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="support/"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h2>Edit Customer</h2>
                    </div>
                    <div class="col-md-6 text-right">
                        <a href="customer_view.php?id=<?php echo $customer_id; ?>" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> Back to Details
                        </a>
                    </div>
                </div>

                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if ($success): ?>
                    <div class="alert alert-success">
                        Customer information updated successfully.
                    </div>
                <?php endif; ?>

                <form method="POST" action="">
                    <div class="row">
                        <div class="col-md-8">
                            <!-- Basic Information -->
                            <div class="card mb-4">
                                <div class="card-header">
                                    <h5 class="mb-0">Basic Information</h5>
                                </div>
                                <div class="card-body">
                                    <div class="form-group">
                                        <label>Username</label>
                                        <input type="text" class="form-control" value="<?php echo htmlspecialchars($customer['username']); ?>" readonly>
                                        <small class="form-text text-muted">Username cannot be changed</small>
                                    </div>
                                    <div class="form-group">
                                        <label>Full Name *</label>
                                        <input type="text" name="full_name" class="form-control" value="<?php echo htmlspecialchars($customer['full_name']); ?>" required>
                                    </div>
                                    <div class="form-group">
                                        <label>Email *</label>
                                        <input type="email" name="email" class="form-control" value="<?php echo htmlspecialchars($customer['email']); ?>" required>
                                    </div>
                                    <div class="form-group">
                                        <label>Phone *</label>
                                        <input type="text" name="phone" class="form-control" value="<?php echo htmlspecialchars($customer['phone']); ?>" required>
                                    </div>
                                    <div class="form-group">
                                        <label>Status *</label>
                                        <select name="status" class="form-control" required>
                                            <option value="active" <?php echo $customer['status'] === 'active' ? 'selected' : ''; ?>>Active</option>
                                            <option value="inactive" <?php echo $customer['status'] === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                            <option value="suspended" <?php echo $customer['status'] === 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- KYC Details -->
                            <div class="card mb-4">
                                <div class="card-header">
                                    <h5 class="mb-0">KYC Details</h5>
                                </div>
                                <div class="card-body">
                                    <div class="form-group">
                                        <label>PAN Number</label>
                                        <input type="text" name="pan_number" class="form-control" value="<?php echo htmlspecialchars($customer['pan_number'] ?? ''); ?>">
                                    </div>
                                    <div class="form-group">
                                        <label>ID Proof Type</label>
                                        <select name="id_proof_type" class="form-control">
                                            <option value="">Select ID Proof Type</option>
                                            <option value="aadhar" <?php echo ($customer['id_proof_type'] ?? '') === 'aadhar' ? 'selected' : ''; ?>>Aadhar Card</option>
                                            <option value="passport" <?php echo ($customer['id_proof_type'] ?? '') === 'passport' ? 'selected' : ''; ?>>Passport</option>
                                            <option value="driving_license" <?php echo ($customer['id_proof_type'] ?? '') === 'driving_license' ? 'selected' : ''; ?>>Driving License</option>
                                            <option value="voter_id" <?php echo ($customer['id_proof_type'] ?? '') === 'voter_id' ? 'selected' : ''; ?>>Voter ID</option>
                                        </select>
                                    </div>
                                    <div class="form-group">
                                        <label>ID Proof Number</label>
                                        <input type="text" name="id_proof_number" class="form-control" value="<?php echo htmlspecialchars($customer['id_proof_number'] ?? ''); ?>">
                                    </div>
                                    <div class="form-group">
                                        <label>KYC Status</label>
                                        <select name="kyc_status" class="form-control">
                                            <option value="pending" <?php echo ($customer['kyc_status'] ?? '') === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                            <option value="verified" <?php echo ($customer['kyc_status'] ?? '') === 'verified' ? 'selected' : ''; ?>>Verified</option>
                                            <option value="rejected" <?php echo ($customer['kyc_status'] ?? '') === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- Banking Details -->
                            <div class="card mb-4">
                                <div class="card-header">
                                    <h5 class="mb-0">Banking Details</h5>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Bank Name</label>
                                                <input type="text" name="bank_name" class="form-control" value="<?php echo htmlspecialchars($customer['bank_name'] ?? ''); ?>">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Account Number</label>
                                                <input type="text" name="account_number" class="form-control" value="<?php echo htmlspecialchars($customer['account_number'] ?? ''); ?>">
                                            </div>
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Account Type</label>
                                                <select name="account_type" class="form-control">
                                                    <option value="">Select Account Type</option>
                                                    <option value="savings" <?php echo ($customer['account_type'] ?? '') === 'savings' ? 'selected' : ''; ?>>Savings</option>
                                                    <option value="current" <?php echo ($customer['account_type'] ?? '') === 'current' ? 'selected' : ''; ?>>Current</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>IFSC Code</label>
                                                <input type="text" name="ifsc_code" class="form-control" value="<?php echo htmlspecialchars($customer['ifsc_code'] ?? ''); ?>">
                                            </div>
                                        </div>
                                    </div>
                                    <div class="form-group">
                                        <label>Branch Name</label>
                                        <input type="text" name="branch_name" class="form-control" value="<?php echo htmlspecialchars($customer['branch_name'] ?? ''); ?>">
                                    </div>
                                    <hr>
                                    <h6>UPI Details</h6>
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>UPI ID</label>
                                                <input type="text" name="upi_id" class="form-control" value="<?php echo htmlspecialchars($customer['upi_id'] ?? ''); ?>">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>UPI Number</label>
                                                <input type="text" name="upi_number" class="form-control" value="<?php echo htmlspecialchars($customer['upi_number'] ?? ''); ?>">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>UPI App</label>
                                                <select name="upi_app" class="form-control">
                                                    <option value="">Select UPI App</option>
                                                    <option value="gpay" <?php echo ($customer['upi_app'] ?? '') === 'gpay' ? 'selected' : ''; ?>>Google Pay</option>
                                                    <option value="phonepe" <?php echo ($customer['upi_app'] ?? '') === 'phonepe' ? 'selected' : ''; ?>>PhonePe</option>
                                                    <option value="paytm" <?php echo ($customer['upi_app'] ?? '') === 'paytm' ? 'selected' : ''; ?>>Paytm</option>
                                                    <option value="bhim" <?php echo ($customer['upi_app'] ?? '') === 'bhim' ? 'selected' : ''; ?>>BHIM</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-md-4">
                            <!-- Quick Actions -->
                            <div class="card mb-4">
                                <div class="card-header">
                                    <h5 class="mb-0">Actions</h5>
                                </div>
                                <div class="card-body">
                                    <button type="submit" class="btn btn-primary btn-block">
                                        <i class="fas fa-save mr-2"></i> Save Changes
                                    </button>
                                    <a href="customer_view.php?id=<?php echo $customer_id; ?>" class="btn btn-secondary btn-block">
                                        <i class="fas fa-times mr-2"></i> Cancel
                                    </a>
                                </div>
                            </div>

                            <!-- Additional Information -->
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="mb-0">Additional Information</h5>
                                </div>
                                <div class="card-body">
                                    <p class="mb-1"><strong>Created:</strong></p>
                                    <p class="text-muted"><?php echo date('M d, Y H:i', strtotime($customer['created_at'])); ?></p>
                                    
                                    <p class="mb-1"><strong>Last Updated:</strong></p>
                                    <p class="text-muted"><?php echo $customer['updated_at'] ? date('M d, Y H:i', strtotime($customer['updated_at'])) : 'Never'; ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 