<?php
require_once '../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    header('Location: ../index.php');
    exit;
}

// Get customer ID
$customer_id = intval($_GET['id'] ?? 0);
if (!$customer_id) {
    header('Location: customers.php');
    exit;
}

// Get customer data
$db = Database::getInstance();
$query = "SELECT username, full_name FROM users WHERE id = ? AND role = 'user'";
$stmt = $db->getConnection()->prepare($query);
$stmt->bind_param("i", $customer_id);
$stmt->execute();
$customer = $stmt->get_result()->fetch_assoc();

if (!$customer) {
    header('Location: customers.php');
    exit;
}

$errors = [];
$success = false;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $new_password = trim($_POST['new_password'] ?? '');
    $confirm_password = trim($_POST['confirm_password'] ?? '');

    // Validate input
    if (empty($new_password)) {
        $errors[] = "New password is required";
    } elseif (strlen($new_password) < 8) {
        $errors[] = "Password must be at least 8 characters long";
    }

    if ($new_password !== $confirm_password) {
        $errors[] = "Passwords do not match";
    }

    // If no errors, proceed with update
    if (empty($errors)) {
        try {
            // Hash the new password
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);

            // Update the password
            $updateQuery = "UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?";
            $stmt = $db->getConnection()->prepare($updateQuery);
            $stmt->bind_param("si", $hashed_password, $customer_id);
            $stmt->execute();

            $success = true;
        } catch (Exception $e) {
            $errors[] = "An error occurred while updating the password: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Change Customer Password</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            padding: 10px 15px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
            text-decoration: none;
        }
        .main-content {
            padding: 20px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <div class="text-center text-white mb-4">
                    <h4><?php echo BRAND_NAME; ?></h4>
                    <p>Admin Panel</p>
                </div>
                <a href="dashboard.php"><i class="fas fa-tachometer-alt mr-2"></i> Dashboard</a>
                <a href="customers.php" class="active"><i class="fas fa-users mr-2"></i> Customers</a>
                <a href="pins/"><i class="fas fa-key mr-2"></i> PIN Management</a>
                <a href="reports/"><i class="fas fa-chart-bar mr-2"></i> Reports</a>
                <a href="support/"><i class="fas fa-headset mr-2"></i> Support</a>
                <a href="../logout.php"><i class="fas fa-sign-out-alt mr-2"></i> Logout</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h2>Change Customer Password</h2>
                    </div>
                    <div class="col-md-6 text-right">
                        <a href="customer_view.php?id=<?php echo $customer_id; ?>" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> Back to Details
                        </a>
                    </div>
                </div>

                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if ($success): ?>
                    <div class="alert alert-success">
                        Password has been updated successfully.
                    </div>
                <?php endif; ?>

                <div class="row">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Change Password for <?php echo htmlspecialchars($customer['full_name']); ?></h5>
                            </div>
                            <div class="card-body">
                                <form method="POST" action="">
                                    <div class="form-group">
                                        <label>Username</label>
                                        <input type="text" class="form-control" value="<?php echo htmlspecialchars($customer['username']); ?>" readonly>
                                    </div>
                                    <div class="form-group">
                                        <label>New Password *</label>
                                        <input type="password" name="new_password" class="form-control" required>
                                        <small class="form-text text-muted">Password must be at least 8 characters long</small>
                                    </div>
                                    <div class="form-group">
                                        <label>Confirm New Password *</label>
                                        <input type="password" name="confirm_password" class="form-control" required>
                                    </div>
                                    <div class="form-group mb-0">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-key mr-2"></i> Change Password
                                        </button>
                                        <a href="customer_view.php?id=<?php echo $customer_id; ?>" class="btn btn-secondary">
                                            <i class="fas fa-times mr-2"></i> Cancel
                                        </a>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Password Guidelines</h5>
                            </div>
                            <div class="card-body">
                                <ul class="mb-0">
                                    <li>Password must be at least 8 characters long</li>
                                    <li>It's recommended to use a combination of:
                                        <ul>
                                            <li>Uppercase letters (A-Z)</li>
                                            <li>Lowercase letters (a-z)</li>
                                            <li>Numbers (0-9)</li>
                                            <li>Special characters (!@#$%^&*)</li>
                                        </ul>
                                    </li>
                                    <li>Avoid using easily guessable information such as:
                                        <ul>
                                            <li>Username</li>
                                            <li>Date of birth</li>
                                            <li>Common words or phrases</li>
                                        </ul>
                                    </li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 