<?php
require_once '../../config/config.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Get POST data
$user_id = intval($_POST['user_id'] ?? 0);
$bank_name = trim($_POST['bank_name'] ?? '');
$account_number = trim($_POST['account_number'] ?? '');
$account_type = trim($_POST['account_type'] ?? '');
$ifsc_code = trim($_POST['ifsc_code'] ?? '');
$branch_name = trim($_POST['branch_name'] ?? '');
$upi_id = trim($_POST['upi_id'] ?? '');
$upi_number = trim($_POST['upi_number'] ?? '');
$upi_app = trim($_POST['upi_app'] ?? '');

if (!$user_id || !$bank_name || !$account_number || !$account_type || !$ifsc_code) {
    echo json_encode(['success' => false, 'message' => 'Missing required banking details']);
    exit;
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Start transaction
    if (!$connection->begin_transaction()) {
        throw new Exception("Failed to start transaction");
    }

    // Check if banking details exist
    $query = "SELECT id FROM banking_details WHERE user_id = ?";
    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare query: " . $connection->error);
    }
    
    $stmt->bind_param("i", $user_id);
    if (!$stmt->execute()) {
        throw new Exception("Failed to check existing banking details: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        // Update existing banking details
        $query = "UPDATE banking_details SET 
                 bank_name = ?, account_number = ?, account_type = ?, 
                 ifsc_code = ?, branch_name = ?, upi_id = ?, 
                 upi_number = ?, upi_app = ? 
                 WHERE user_id = ?";
    } else {
        // Insert new banking details
        $query = "INSERT INTO banking_details 
                 (bank_name, account_number, account_type, ifsc_code, 
                  branch_name, upi_id, upi_number, upi_app, user_id) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    }

    $stmt = $connection->prepare($query);
    if (!$stmt) {
        throw new Exception("Failed to prepare query: " . $connection->error);
    }
    
    $stmt->bind_param("ssssssssi", 
        $bank_name, $account_number, $account_type, 
        $ifsc_code, $branch_name, $upi_id, 
        $upi_number, $upi_app, $user_id
    );
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to update banking details: " . $stmt->error);
    }

    // Commit transaction
    if (!$connection->commit()) {
        throw new Exception("Failed to commit transaction");
    }

    echo json_encode([
        'success' => true,
        'message' => 'Banking details updated successfully'
    ]);

} catch (Exception $e) {
    if (isset($connection) && $connection->ping()) {
        $connection->rollback();
    }
    
    error_log("Banking Details Update Error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while updating banking details: ' . $e->getMessage()
    ]);
} 