<?php
require_once __DIR__ . '/../../config/config.php';

// Skip authentication when running from command line
if (php_sapi_name() !== 'cli') {
    // Check if user is logged in and is admin
    $user = new User();
    if (!$user->isLoggedIn() || !$user->isAdmin()) {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        exit;
    }
}

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    // Start transaction
    $connection->begin_transaction();
    
    // Array of SQL queries to modify the schema
    $queries = [
        // Add 'expired' status to pins table if not exists
        "ALTER TABLE pins MODIFY COLUMN status ENUM('created', 'transferred', 'activated', 'expired') DEFAULT 'created'",
        
        // Add updated_at column to pins table if not exists
        "ALTER TABLE pins ADD COLUMN IF NOT EXISTS updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"
    ];
    
    $success = true;
    $messages = [];
    
    // Execute each query
    foreach ($queries as $query) {
        try {
            $connection->query($query);
            $messages[] = "Successfully executed: $query";
        } catch (Exception $e) {
            // Ignore errors about column/status already existing
            if (!strpos($e->getMessage(), 'Duplicate') && !strpos($e->getMessage(), 'already exists')) {
                throw $e;
            }
            $messages[] = "Notice: " . $e->getMessage();
        }
    }
    
    // Commit transaction
    $connection->commit();
    
    if (php_sapi_name() === 'cli') {
        echo implode("\n", $messages) . "\n";
        echo "Schema updated successfully\n";
    } else {
        echo json_encode([
            'success' => true,
            'message' => 'Schema updated successfully',
            'details' => $messages
        ]);
    }

} catch (Exception $e) {
    // Rollback transaction on error
    if (isset($connection)) {
        $connection->rollback();
    }
    
    $error = 'Error updating schema: ' . $e->getMessage();
    
    if (php_sapi_name() === 'cli') {
        echo $error . "\n";
        exit(1);
    } else {
        echo json_encode([
            'success' => false,
            'message' => $error
        ]);
    }
} 