<?php
require_once '../../config/config.php';

// Check if user is logged in and is admin
$user = new User();
if (!$user->isLoggedIn() || !$user->isAdmin()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

// Get customer ID
$customer_id = isset($_POST['id']) ? intval($_POST['id']) : 0;

// Validate input
if (!$customer_id) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid customer ID']);
    exit;
}

try {
    $db = Database::getInstance();
    $db->getConnection()->begin_transaction();

    // First, check if the customer exists and is not an admin
    $checkQuery = "SELECT role FROM users WHERE id = ?";
    $stmt = $db->getConnection()->prepare($checkQuery);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        throw new Exception("Customer not found");
    }
    
    $user = $result->fetch_assoc();
    if ($user['role'] === 'admin') {
        throw new Exception("Cannot delete admin users");
    }

    // Delete related records first (due to foreign key constraints)
    
    // 1. Delete banking details
    $query = "DELETE FROM banking_details WHERE user_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    // 2. Delete user profiles
    $query = "DELETE FROM user_profiles WHERE user_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    // 3. Delete ticket messages
    $query = "DELETE FROM ticket_messages WHERE sender_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    // 4. Delete support tickets
    $query = "DELETE FROM support_tickets WHERE user_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    // 5. Delete pin transfers
    $query = "DELETE FROM pin_transfers WHERE from_user = ? OR to_user = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("ii", $customer_id, $customer_id);
    $stmt->execute();

    // 6. Delete pins
    $query = "DELETE FROM pins WHERE created_by = ? OR current_owner = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("ii", $customer_id, $customer_id);
    $stmt->execute();

    // 7. Delete genealogy
    $query = "DELETE FROM genealogy WHERE user_id = ? OR parent_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("ii", $customer_id, $customer_id);
    $stmt->execute();

    // 8. Delete earnings
    $query = "DELETE FROM earnings WHERE user_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    // 9. Delete withdrawals
    $query = "DELETE FROM withdrawals WHERE user_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    // 10. Delete user products
    $query = "DELETE FROM user_products WHERE user_id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    // Finally, delete the user
    $query = "DELETE FROM users WHERE id = ?";
    $stmt = $db->getConnection()->prepare($query);
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();

    if ($stmt->affected_rows === 0) {
        throw new Exception("Failed to delete customer");
    }

    $db->getConnection()->commit();
    echo json_encode(['success' => true]);

} catch (Exception $e) {
    $db->getConnection()->rollback();
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} 